/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */

package horb.orb;

import java.util.Hashtable;
import java.util.Date;
import java.net.*;
import java.io.*;


/**
 * item in the thread table
 * 
 */
class ThreadServer extends Thread {
  ThreadStatus ts;		// objectID, startTime, serviceTime...

  Thread thread;		// this
  HORBServer server;		// HORB server for this port
  Skeleton skeleton;		// skeleton
  IOCI ioci;			// IOCI

  ThreadServer(HORBServer server, IOCI ioci) {
    ts = new ThreadStatus();

    this.thread = this;
    ts.startTime = System.currentTimeMillis();
    this.ioci = ioci;
    this.server = server;
    ts.port = server.ss.port;
  }

  /*
   * Called as a thread for each beginning of a connection
   */
  public void run() {
    try {
      if (server.ss.debug)
	ioci.setProperty(IOCI.PROP_DEBUG, null);
      if (ioci.recvConnectServer() == false) {
	close();
	return;
      }

      ts.netAddress = ioci.getAddress();
      if (server.ss.hostnameLookup) {
	ts.clientName = ioci.getHostName();
	if ("localhost".equals(ts.clientName))
	  ts.clientName = ioci.getLocalHostName();
      }

      short req = ioci.recvShort();

      switch (req) {
      case ioci.ORB_CREATE_INSTANCE:
	createInstance();
	break;
      case ioci.ORB_CONNECT_TO_INSTANCE:
	connectToObject();
	break;
      case ioci.ORB_INVITE_TO_INSTANCE:
	invited();
	return;			// don't close the connection
      case ioci.ORB_CLOSE:
	break;
      default:
	server.msg("unknown request " + req);
	break;
      }
    } catch (Exception e) {
      // TODO: something
      server.msg("HORB Service request error: "+e);
      e.printStackTrace();
    } catch (Throwable t) {
      server.msg("Thread killed "+ t);
    }
    server.msg("close connection");
    close();
  }

  /**
   * create a new HORB object.
   */
  protected void createInstance() {
    boolean registered = false;
    ObjectInfo oi = null;
    NetIOCIInfo header = null;

    try {
      try {
	header = ioci.recvCreateObject();
	ts.userName = header.username;
	ts.major_version = header.major_version;
	ts.minor_version = header.minor_version;
      } catch (Exception e) {
	return;
      }

      // check the access control list
      String s = null;
      if (server.acl.checkHostUser_Local(header.className, (IOCIService)ioci) == false
	  || (((s = server.acl.getValue_Local(header.className+".creatable")) != null) && "false".equals(s))) {
	server.msg("no permission: " + header.className + ": " + ts.userName
		   + "@"+ ts.clientName);
	ioci.sendStatus(ioci.STAT_PERMISSION_ERROR);
	ioci.kick();
	return;
      }

      String skeletonClassName = header.className+"_Skeleton";
      server.msg("creating an instance of " + header.className);

      // make it into class descriptor
      skeleton = (Skeleton)Class.forName(skeletonClassName).newInstance();
//      object = skeleton.createObject();

      if (header.major_version != 0 && server.ss.versionCheck
	  && skeleton.getMajorVersion() != header.major_version) {
	server.msg("class major version mismatch: " + skeletonClassName
		   + " = " +skeleton.getMajorVersion()
		   +" but requestor's major = "+header.major_version);
	ioci.sendStatus(ioci.STAT_CLASS_VERSION_MISMATCH);
	ioci.kick();
	return;
      }
      //
      // make unique object ID and register it with the object
      // 
      int x = 13;
      do {
	ts.objectID = "H"+(skeleton.hashCode()+x);
	x = x * x;
      } while (HORBServer.objectTable.containsKey(ts.objectID));
      server.msg("HORB object "+ts.objectID+" has been created");
      ts.threadName = ts.objectID+"-0";
      Thread.currentThread().setName(ts.threadName);
      oi = HORBServer.objectTable.put(header.className, ts.objectID, 
				      skeleton, ts.port, false);
      HORBServer.objectTable.use(oi);
      HORBServer.threadTable.put(this, this);
      server.ss.lastUsedTime = System.currentTimeMillis();
      registered = true;
      //
      ioci.sendStatus(ioci.STAT_NO_ERROR);	// OK ack
      ioci.sendString(ts.objectID);	// return the objectID
      ioci.sendString(ts.threadName);	// return the thread name
      ioci.kick();

      // enter the call waiting loop
      MethodCallLoop();

      // clean up
      HORBServer.objectTable.unuse(oi);
      HORBServer.threadTable.remove(this);
      server.msg("thread ended from createInstance");
    } catch (Exception e) {
      try {
	ioci.sendStatus(ioci.STAT_NO_SUCH_OBJECT);// BAD ack
	ioci.kick();
      } catch (Exception e1) {}
      server.msg("instance creation failed - "+e.toString());
      if (registered) {
	HORBServer.objectTable.unuse(oi);
	HORBServer.threadTable.remove(this);
      }
      return;			
      // and connection will be closed by the caller
    }
  }

  /**
   * connect to a registered object.
   */
  protected void connectToObject() {
    short stat = ioci.STAT_NO_ERROR;
    ObjectInfo oi = null;
    boolean registered = false;
    NetIOCIInfo header = null;

    try {
      header = ioci.recvConnectObject();
      ts.objectID = header.objectID;
      ts.userName = header.username;
      ts.major_version = header.major_version;
      ts.minor_version = header.minor_version;
    } catch (IOException e) {
      return;
    }
    try {
      server.msg("connecting to object " + ts.objectID);
      // lookup object in objectTable
      oi = (ObjectInfo)HORBServer.objectTable.get(ts.objectID);
      if (oi == null || oi.os.stop || (skeleton = oi.getSkeleton()) == null) {
	ioci.sendStatus(ioci.STAT_NO_SUCH_OBJECT);
	ioci.kick();
	server.msg("no such object " + ts.objectID);
	return;
      }
      if (oi.os.className.equals(header.className) == false) {
	server.msg("warning: className mismatch for " + ts.objectID);
	server.msg("  registered: "+oi.os.className+" requested: "+header.className);
	ioci.sendStatus(ioci.STAT_NO_SUCH_OBJECT);
	ioci.kick();
	return;
      }
      if ((oi.os.port == 0 || oi.os.port == ts.port) &&
	  server.acl.checkHostUser_Local(oi.os.className, (IOCIService)ioci) == false) {
	ioci.sendStatus(ioci.STAT_PERMISSION_ERROR);
	ioci.kick();
	server.msg("no permission: " + oi.os.className+"("+ts.objectID
		   + "): " + ts.userName + "@"+ ts.clientName);
	return;
      }

      if (header.major_version != 0 && server.ss.versionCheck
	  && skeleton.getMajorVersion() != header.major_version) {
	ioci.sendStatus(ioci.STAT_CLASS_VERSION_MISMATCH);
	ioci.kick();
	server.msg("class major version mismatch: " + oi.os.className
		   + "_Skeleton = " +skeleton.getMajorVersion()
		   +" but requestor's major = "+header.major_version);
	return;
      }

      ts.threadName = ts.objectID+"-"+oi.os.numConnect;
      server.msg("HORB object "+ts.objectID+" has been connected");
      Thread.currentThread().setName(ts.threadName);
      HORBServer.objectTable.use(oi);
      HORBServer.threadTable.put(this, this);
      registered = true;
      server.ss.lastUsedTime = System.currentTimeMillis();

      ioci.sendStatus(ioci.STAT_NO_ERROR);
      ioci.sendString(ts.threadName);
      ioci.kick();

      // enter the call waiting loop
      MethodCallLoop();

      // cleanup
      HORBServer.objectTable.unuse(oi);
      HORBServer.threadTable.remove(this);
      server.msg("thread ended from connectToObject");
    } catch (Exception f) {
      if (registered) {
	HORBServer.objectTable.unuse(oi);
	HORBServer.threadTable.remove(this);
      }
      return;
    }
  }

  void invited() {
    NetIOCIInfo header = null;

    try {
      header = ioci.recvInvite();

      Proxy proxy = (Proxy)Class.forName(header.className+"_Proxy").newInstance();
      HorbURL clientURL = new HorbURL(header.clientURL); 
      server.msg(header.threadName+"is invited to "+header.clientURL);
      // start connecting to server
      proxy._invited(ioci, clientURL, header.username, header.passwd);
      ThreadServer ts = HORBServer.threadTable.getThreadServer(header.threadName);
      ts.skeleton.invited(proxy);
/*
      ioci.sendStatus(ioci.STAT_NO_ERROR);
      ioci.kick();
*/
    } catch (Exception e) {
      server.msg("ThreadServer.invited "+e);
      try {
	ioci.sendStatus(ioci.STAT_NO_SUCH_OBJECT);
	ioci.kick();
	ioci.release();
      } catch (Exception e0) {}
    }
  }

  /**
   * method call loop. this runs in a thread context.
   */
  protected void MethodCallLoop() {
    boolean constructorFail;
    try {
      for (;;) {
	server.msg("Waiting for method call");
	short preamble = ioci.recvPreamble();
	if (preamble == -1 || preamble == ioci.ORB_CLOSE)
	  break;
	if (preamble != ioci.PREAMBLE) {
	  server.msg("bad preamble - "+preamble);
	  break;
	}
	ts.classNo = ioci.acceptClassNo();
	ts.methodNo = ioci.acceptMethod();
	if (ts.classNo < 0 || ts.methodNo < 0)
	  break;
	server.msg("classNo = "+ts.classNo+" method no = "+ts.methodNo);

	// recording
	ts.requests++;
	//
	// call the object
	//
	constructorFail = skeleton.dispatch(ioci, ts.classNo, ts.methodNo);
	ts.lastClassNo = ts.classNo;
	ts.lastMethodNo = ts.methodNo;
	ts.classNo = 0;
	ts.methodNo = 0;
	if (constructorFail)
	  break;
      }
    } catch (IOException e) {
      // connection lost
      server.msg("disconnected - "+e.toString());
    } catch (Throwable t) {
      // thread killed
      server.msg("thread killed - "+t.toString());
      // exit methodcallloop for cleanup
    }


    // run finalize() of the HORB object.
    skeleton.threadEndHook();
  }

  public void close() {
    ioci.release();
    ioci = null;
  }

  /**
   * return a clone of thread status 
   */
  ThreadStatus getThreadStatus() {
    return ts.get();
  }

  final boolean objectMatch(String objectID) {
    return objectID.equals(this.ts.objectID);
  }
}
