/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */

package horb.orb;

import java.net.InetAddress;
import java.io.*;
import horb.tools.util.*;

/**
 * container class to carry status of threads.
 */
public class ThreadStatus implements Cloneable {
  /**
   * name of this HORB thread. You can suspend, resume, stop this thread
   * with class HORBAgent.
   */
  public String threadName;

  /**
   * object ID of this thread.
   */
  public String objectID;

  /**
   * major version number of peer proxy class.
   */
  public short major_version;

  /**
   * minor version number of peer proxy class.
   */
  public short minor_version;

  /**
   * threads' priority
   */
  public int priority;

  /**
   * port of this thread
   */
  public int port;

  /**
   * client's network address (e.g. IP address) in network byte order. 
   * This can be a 64bit IP address.
   */
  public byte[] netAddress;

  /**
   * client's hostname. this parameter can be obtained if clientName parameter
   * is true when HORBAgent.getServerStatus() is called. This option
   * causes the operation quite slow. However, we don't have any means
   * to get hostname from network address, because InetAddress.getNameByAddr()
   * is a private method (;-<).
   * @param clientName return client's hostname in ThreadStatus.clinetName
   */
  public String clientName;

  /**
   * client's username. If null, it is anonymous access.
   */
  public String userName;

  /**
   * start time of this thread. Milli sec since epoch.
   */
  public long startTime;

  /**
   * the time of request. Milli sec since epoch.
   * currentTime-startTime shows life time of this thread.
   */
  public long currentTime;

  /**
   * number of calls for methods.
   */
  public int requests;

  /**
   * method number of the currently being called method. Zero if no method
   * is under execution currently. You can find method number in 
   * Class_Skeleton.dispatch().
   */
  public short methodNo;

  /**
   * class number of the currently being called method. This value is 
   * corresponding to the methodNo. By using classNo and methodNo,
   * you can specify what method is called now.
   * If a method of a class is under 
   * execution and the class is the 3rd subclass of java.class.Object,
   * the class has class number 3.
   */
  public short classNo;

  /**
   * method number of the lastly called method. Zero if no method
   * has been executed.
   */
  public short lastMethodNo;

  /**
   * class number of the lastly called method.
   */
  public short lastClassNo;

  public synchronized ThreadStatus get() {
    try {
      return (ThreadStatus)this.clone();
    } catch (Exception e) {}
    return null;
  }

  public void list(PrintStream ps) {
    FormatDate startTime = new FormatDate(this.startTime);
    
    ps.println("\t\tthread name: "+threadName);
    ps.println("\t\tobjectID: "+objectID);
    ps.println("\t\tclient proxy version: "+major_version+"."+minor_version);
    ps.println("\t\tthread priority: "+priority);

    ps.println("\t\tstart time: "+startTime.getYMDHMS());
    ps.println("\t\tport: "+port);
    ps.println("\t\tclient hostname: "+clientName);
    String user = userName;
    if (user == null)
      user = "anonymous";
    ps.println("\t\tclient username: "+user);
    ps.print("\t\tclient network address: ");
    for (int i = 0; i < netAddress.length; i++) {
      ps.print(netAddress[i] & 0xff);
      if (i+1 < netAddress.length)
	ps.print(".");
    }
    ps.println();
    ps.println("\t\tnumber of method calls: "+requests);
    ps.print("\t\tlastly called method: ");
    if (lastMethodNo == 0)
      ps.println("none");
    else
      ps.println("classNo = "+lastClassNo+" methodNo = "+lastMethodNo);
    ps.print("\t\tcurrently called method: ");
    if (methodNo == 0)
      ps.println("none");
    else
      ps.println("classNo = "+classNo+" methodNo = "+methodNo);
  }
}
