/*
 * $Header$
 *
 * Copyright (C) 1995, 1996 HIRANO Satoshi
 *
 * Permission to use, copy, modify and redistribution this software in
 * whole and in part, for evaluation or research purposes and without fee
 * is hereby granted provided this copyright notice.
 * See CopyrightAndLicensing.txt for licensing condition.
 */

package horb.orb;

import java.util.*;
import java.net.*;

/**
 * Thread table
 *
 * records every thread activity created by reqeusts
 */
class ThreadTable extends Hashtable {

  final synchronized ThreadServer get(ThreadServer ts) {
    return (ThreadServer)super.get(ts);
  }

  synchronized final ThreadServer getThreadServer(String threadName) throws HORBException {
    Enumeration e = elements();
    while (e.hasMoreElements()) {
      ThreadServer t = (ThreadServer)e.nextElement();
      if (threadName.equals(t.ts.threadName)) {
	return t;
      }
    }
    throw new HORBException("thread not found");
  }

  /**
   * terminate a thread.
   */
  final synchronized void stop(String threadName) throws HORBException {
    ThreadServer t = getThreadServer(threadName);
    t.stop();
  }
    
  /**
   * suspend a thread.
   */
  final synchronized void suspend(String threadName) throws HORBException {
    ThreadServer t = getThreadServer(threadName);
    t.suspend();
  }

  /**
   * resume a thread.
   */
  final synchronized void resume(String threadName) throws HORBException {
    ThreadServer t = getThreadServer(threadName);
    t.suspend();
  }

  /**
   * set priority of a thread.
   * @param newPriority new priority. Value should be within the range
   * Thread.MIN_PRIORTY,  Thread.MAX_PRIORITY.
   */
  final synchronized void setPriority(String threadName, int newPriority) throws HORBException, IllegalArgumentException {
    ThreadServer t = getThreadServer(threadName);
    t.setPriority(newPriority);
  }

  /**
   * get priority of a thread.
   */
  final synchronized int getPriority(String threadName) throws HORBException {
    ThreadServer t = getThreadServer(threadName);
    return t.getPriority();
  }


  /**
   * returns status of threads of all HORB object.
   * This method is not synchronized because InetAddress.getHostName()
   * is very time consuming. 
   */
  final synchronized ThreadStatus[] getThreadStatus() {
    int size;
    long now = System.currentTimeMillis();

    // first put threads of the objectID into the vector
    size = size();
    ThreadStatus ts[] = new ThreadStatus[size];
    Enumeration e = elements();
    for (int i = 0; i < size && e.hasMoreElements(); i++) {
      ThreadServer t = (ThreadServer)e.nextElement();
      ts[i] = t.ts;
      ts[i].priority = t.getPriority();
      ts[i].currentTime = now;
    }
    return ts;
  }

  /**
   * returns status of threads of a HORB object.
   */
  final synchronized ThreadStatus[] getThreadStatus(String objectID) {
    int size = 0;
    long now = System.currentTimeMillis();

    // first put threads of the objectID into the vector
    Enumeration e = elements();
    ThreadStatus ts[] = getThreadStatusSub(e, objectID, 0);
    for (int i = ts.length -1; i >= 0; --i)
      ts[i].currentTime = now;
    return ts;
  }

  private final synchronized ThreadStatus[] getThreadStatusSub(Enumeration e, String objectID, int num) {
    boolean match = false;
    ThreadStatus[] ts = null;
    if (e.hasMoreElements()) {
      ThreadServer t = (ThreadServer)e.nextElement();
      if (t.ts.objectID.equals(objectID)) {
	match = true;
	num++;
      }
      ts = getThreadStatusSub(e, objectID, num);
      if (match) {
	ts[num-1] = t.ts;
	ts[num-1].priority = t.getPriority();
      }
    } else
      ts = new ThreadStatus[num];
    return ts;
  }
}
