/* $Id: nonlwin.c,v 1.1 1995/04/13 16:25:49 pturner Exp pturner $
 *
 * non linear curve fitting
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/DialogS.h>
#include <Xm/Frame.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/ScrolledW.h>

#include "globals.h"
#include "protos.h"
#include "motifinc.h"

/* info strings
info = 0  improper input parameters.
info = 1  algorithm estimates that the relative error in the sum of squares is at most tol.
info = 2  algorithm estimates that the relative error between x and the solution is at most tol.
info = 3  conditions for info = 1 and info = 2 both hold.
info = 4  fvec is orthogonal to the columns of the jacobian to machine precision.
info = 5  number of calls to fcn has reached or exceeded the asked value (OK).
info = 6  tol is too small. no further reduction in the sum of squares is possible.
info = 7  tol is too small. no further improvement in the approximate solution x is possible.
*/

extern double nonl_parms[];

static void do_nonl_proc(Widget w, XtPointer client_data, XtPointer call_data);

#define NONL_CANCEL 0
#define NONL_ACCEPT 1

static Widget nonl_frame;
static Widget nonl_panel;
Widget nonl_formula_item;
SetChoiceItem nonl_set_item;
static Widget *nonl_load_item;
static Widget nonl_param_item[MAXPARM];
static Widget nonl_constr_item[MAXPARM];
static Widget nonl_lowb_item[MAXPARM];
static Widget nonl_uppb_item[MAXPARM];
static Widget nonl_tol_item;
static Widget nonl_nparm_item;
static Widget nonl_autol_item;
static Widget nonl_npts_item;
static Widget nonl_start_item, nonl_stop_item; 
Widget nonl_fload_rc;
static void do_nonl_proc(Widget w, XtPointer client_data, XtPointer call_data);
static void do_nonl_toggle(Widget w, XtPointer client_data, XtPointer call_data);
static void do_constr_toggle (Widget w, XtPointer client_data, XtPointer call_data);
static void destroy_nonl_frame(Widget w, XtPointer client_data, XtPointer call_data);

/* whether or not to use constraints and if yes, low & upper bounds */
int  nonl_constr[MAXPARM];
double nonl_lowb[MAXPARM];
double nonl_uppb[MAXPARM];

int nsteps;
int nlsetno;
int nlloadset = -1;

/* ARGSUSED */
void create_nonl_frame(Widget w, XtPointer client_data, XtPointer call_data)
{
    int i;
    int x, y;
    Widget sw, fr, rc, rc1, lab, fitbut[3], but1[4];
    set_wait_cursor();
    if (nonl_frame == NULL) {
	char *fitlabel[3];
	char *label1[4];

	fitlabel[0] = "5 steps";
	fitlabel[1] = "20 steps";
	fitlabel[2] = "100 steps";
	
	label1[0] = "Load";
	label1[1] = "Accept";
	label1[2] = "Cancel";
	label1[3] = "Help";

	XmGetPos(app_shell, 0, &x, &y);
	nonl_frame = XmCreateDialogShell(app_shell, "Non-linear curve fitting", NULL, 0);
	handle_close(nonl_frame);
	XtVaSetValues(nonl_frame, XmNx, x, XmNy, y, NULL);
	nonl_panel = XmCreateForm(nonl_frame, "nonl_frame_rc", NULL, 0);
	fr = XmCreateFrame(nonl_panel, "nonl_frame", NULL, 0);
	rc = XmCreateRowColumn(fr, "nonl_rc", NULL, 0);

        nonl_set_item  = CreateSetSelector(rc, "Apply to set:",
                                    SET_SELECT_ACTIVE,
                                    FILTER_SELECT_NONE,
                                    GRAPH_SELECT_CURRENT,
                                    SELECTION_TYPE_SINGLE);

	nonl_formula_item = (Widget) CreateTextItem2(rc, 35, "Function:");
	xv_setstr(nonl_formula_item, "y = ");

	rc1 = XmCreateRowColumn(rc, "nonl_rc", NULL, 0);
	XtVaSetValues(rc1, XmNorientation, XmHORIZONTAL, NULL);
	nonl_nparm_item = CreateTextItem2(rc1, 5, "# of parameters:");
	nonl_tol_item = CreateTextItem2(rc1, 10, "Tolerance:");
	xv_setstr(nonl_tol_item, "0.01");
	XtManageChild(rc1);
	
	XtManageChild(rc);
	XtManageChild(fr);
	
	sw = XtVaCreateManagedWidget("sw",
				 xmScrolledWindowWidgetClass, nonl_panel,
				     XmNheight, 250,
				     XmNscrollingPolicy, XmAUTOMATIC,
				     NULL);

	rc = XmCreateRowColumn(sw, "rc", NULL, 0);


	for (i = 0; i < MAXPARM; i++) {
	    rc1 = XmCreateRowColumn(rc, "rc1", NULL, 0);
	    XtVaSetValues(rc1, XmNorientation, XmHORIZONTAL, NULL);
	    sprintf(buf, "A%1d: ", i);
	    nonl_param_item[i] = CreateTextItem2(rc1, 10, buf);
	    /* a non-zero initial values work better */
	    xv_setstr(nonl_param_item[i], "1.0");

	    nonl_constr_item[i] = XmCreateToggleButton(rc1, "Bounds:", NULL, 0);
	    XtAddCallback(nonl_constr_item[i], XmNvalueChangedCallback, 
	    	    	    (XtCallbackProc) do_constr_toggle, (XtPointer) i);
	    XtManageChild(nonl_constr_item[i]);
	    nonl_constr[i] = FALSE;

	    nonl_lowb_item[i] = CreateTextItem2(rc1, 6, "");
	    xv_setstr(nonl_lowb_item[i], "1.0");
	    XtSetSensitive(nonl_lowb_item[i], False);
	    
	    sprintf(buf, "< A%1d < ", i);
	    lab = XmCreateLabel(rc1, buf, NULL, 0);
    	    XtManageChild(lab);

	    nonl_uppb_item[i] = CreateTextItem2(rc1, 6, "");
	    xv_setstr(nonl_uppb_item[i], "1.0");
	    XtSetSensitive(nonl_uppb_item[i], False);
	    XtManageChild(rc1);
	}

	XtManageChild(rc);
	XtVaSetValues(sw,
		      XmNworkWindow, rc,
		      NULL);

	rc = XmCreateRowColumn(nonl_panel, "rc", NULL, 0);
	
	CreateCommandButtons(rc, 3, fitbut, fitlabel);
	for (i = 0; i < 3; i++){
	     XtAddCallback(fitbut[i], XmNactivateCallback,
	     		   (XtCallbackProc) do_nonl_proc, (XtPointer) (i + 1));
	}
	
	XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, rc, NULL);
	
	rc1 = XmCreateRowColumn(rc, "nonl_rc", NULL, 0);
	XtVaSetValues(rc1, XmNorientation, XmHORIZONTAL, NULL);
	nonl_load_item = CreatePanelChoice(rc1,
				   "Load:",
				   4,
				   "Fitted values",
				   "Residuals",
				   "Function",
				   NULL, NULL);

	for (i = 0; i < 3; i++) {
	    XtAddCallback(nonl_load_item[2 + i], XmNactivateCallback,
			(XtCallbackProc) do_nonl_toggle, (XtPointer) i);
	}
	
	nonl_autol_item = XtVaCreateManagedWidget("Autoload",
			    xmToggleButtonWidgetClass, rc1, NULL);
	XtManageChild(rc1);
	
	nonl_fload_rc = XmCreateRowColumn(rc, "nonl_fload_rc", NULL, 0);
	XtVaSetValues(nonl_fload_rc, XmNorientation, XmHORIZONTAL, NULL);
	nonl_start_item = CreateTextItem2(nonl_fload_rc, 10, "Start:");
	nonl_stop_item = CreateTextItem2(nonl_fload_rc, 10, "Stop:");
	nonl_npts_item = CreateTextItem2(nonl_fload_rc, 6, "# of points:");
	XtManageChild(nonl_fload_rc);
	XtSetSensitive(nonl_fload_rc, False);

	CreateCommandButtons(rc, 4, but1, label1);
	XtAddCallback(but1[0], XmNactivateCallback,
		(XtCallbackProc) do_nonl_proc, (XtPointer) 0);
	XtAddCallback(but1[1], XmNactivateCallback,
		(XtCallbackProc) destroy_nonl_frame, (XtPointer) NONL_ACCEPT);
	XtAddCallback(but1[2], XmNactivateCallback,
		(XtCallbackProc) destroy_nonl_frame, (XtPointer) NONL_CANCEL);
	XtAddCallback(but1[3], XmNactivateCallback,
		(XtCallbackProc) HelpCB, (XtPointer) NULL);
	XtManageChild(rc);

	XtVaSetValues(fr,
		      XmNtopAttachment, XmATTACH_FORM,
		      XmNleftAttachment, XmATTACH_FORM,
		      XmNrightAttachment, XmATTACH_FORM,
		      NULL);
	XtVaSetValues(sw,
		      XmNtopAttachment, XmATTACH_WIDGET,
		      XmNtopWidget, fr,
		      XmNleftAttachment, XmATTACH_FORM,
		      XmNrightAttachment, XmATTACH_FORM,
		      XmNbottomAttachment, XmATTACH_WIDGET,
		      XmNbottomWidget, rc,
		      NULL);
	XtVaSetValues(rc,
		      XmNleftAttachment, XmATTACH_FORM,
		      XmNrightAttachment, XmATTACH_FORM,
		      XmNbottomAttachment, XmATTACH_FORM,
		      NULL);

	XtManageChild(nonl_panel);
    }
    XtRaise(nonl_frame);
    unset_wait_cursor();
}

static void do_nonl_toggle (Widget w, XtPointer client_data, XtPointer call_data)
{
    int value = (int) client_data;
    if (value == 2) {
    	XtSetSensitive(nonl_fload_rc, True);
    } else {
    	XtSetSensitive(nonl_fload_rc, False);
    }
}

static void do_constr_toggle (Widget w, XtPointer client_data, XtPointer call_data)
{
    int value = (int) client_data;
    if (XmToggleButtonGetState(nonl_constr_item[value])) {
    	XtSetSensitive(nonl_lowb_item[value], True);
    	XtSetSensitive(nonl_uppb_item[value], True);
    	nonl_constr[value] = TRUE;
    } else {
    	XtSetSensitive(nonl_lowb_item[value], False);
    	XtSetSensitive(nonl_uppb_item[value], False);
    	nonl_constr[value] = FALSE;
    }
}

/* ARGSUSED */
static void do_nonl_proc(Widget w, XtPointer client_data, XtPointer call_data)
{
    int i, n, npar, npts = 0, info;
    int loadto;
    double tol;
    char fstr[256];
    double *y, *yp;
    double start, stop, delx, *xtmp, *ytmp;
    int value = (int) client_data;
    
    set_wait_cursor();
    curset = nlsetno = GetSelectedSet(nonl_set_item);
    if (curset == SET_SELECT_ERROR) {
    	errwin("No set selected");
    	unset_wait_cursor();
    	return;
    }
    tol = atof((char *) xv_getstr(nonl_tol_item));
    npar = atoi((char *) xv_getstr(nonl_nparm_item));
    strcpy(fstr, (char *) xv_getstr(nonl_formula_item));
    for (i = 0; i < npar; i++) {
/*
 * 	nonl_parms[i] = 1.0; 
 */
	strcpy(buf, (char *) xv_getstr(nonl_param_item[i]));
	if (sscanf(buf, "%lf", &nonl_parms[i]) != 1) {
	    errwin("Invalid input in parameter field");
	    unset_wait_cursor();
	    return;
	}
	if (nonl_constr[i]) {
	    strcpy(buf, (char *) xv_getstr(nonl_lowb_item[i]));
	    if (sscanf(buf, "%lf", &nonl_lowb[i]) != 1) {
	    	errwin("Invalid input in low-bound field");
	    	unset_wait_cursor();
	    	return;
	    }
	    strcpy(buf, (char *) xv_getstr(nonl_uppb_item[i]));
	    if (sscanf(buf, "%lf", &nonl_uppb[i]) != 1) {
	    	errwin("Invalid input in upper-bound field");
	    	unset_wait_cursor();
	    	return;
	    }
	    if ((nonl_parms[i] < nonl_lowb[i]) || (nonl_parms[i] > nonl_uppb[i])) {
	    	errwin("Initial values must be within bounds");
	    	unset_wait_cursor();
	    	return;
	    }
	}
    }
    
    n = getsetlength(cg, nlsetno);
    
    switch (value) {
    	case 1:
    	    nsteps = 5*(n + 1);
    	    break;
    	case 2:
    	    nsteps = 20*(n + 1);
    	    break;
    	case 3:
    	    nsteps = 100*(n + 1);
    	    break;
    	default:
    	    nsteps = 0;
    	    break;
    }
    
    y = gety(cg, nlsetno);
    
    if (nsteps) { /* we are asked to fit */
    	yp = (double *) calloc(n, sizeof(double));
    	if (yp == NULL) {
	    errwin("Memory allocation error, operation cancelled");
	    unset_wait_cursor();
	    return;
    	}
    	for (i = 0; i < n; i++) {
	    yp[i] = y[i];
    	}
    	sprintf(buf, "Fitting: %s\n", fstr);
    	stufftext(buf, 0);
    	sprintf(buf, "Initial guess:\n");
    	stufftext(buf, 0);
    	for (i = 0; i < npar; i++) {
	    sprintf(buf, "\ta%1d = %.9f\n", i, nonl_parms[i]);
	    stufftext(buf, 0);
    	}
    	sprintf(buf, "Tolerance = %.9f\n", tol);
    	stufftext(buf, 0);
    	lmfit(fstr, n, getx(cg, nlsetno),
	      yp, y, npar, tol, nsteps, &info);
    	for (i = 0; i < n; i++) {
	    y[i] = yp[i];
    	}
    	free(yp);
    	for (i = 0; i < npar; i++) {
	    sprintf(buf, "%.8f", nonl_parms[i]);
	    xv_setstr(nonl_param_item[i], buf);
    	}

    	if ((info > 0 && info < 4) || (info == 5)) {
	    sprintf(buf, "Computed values:\n");
	    stufftext(buf, 0);
	    for (i = 0; i < npar; i++) {
		sprintf(buf, "\ta%1d = %.9f\n", i, nonl_parms[i]);
		stufftext(buf, 0);
	    }
    	}


    	if (info >= 0 && info <= 7) {
    	    char *s;
    	    switch (info) {
    	    case 0:
    		s = "Improper input parameters.\n";
    		break;
    	    case 1:
    		s = "Relative error in the sum of squares is at most tol.\n";
    		break;
    	    case 2:
    		s = "Relative error between A and the solution is at most tol.\n";
    		break;
    	    case 3:
    		s = "Relative error in the sum of squares and A and the solution is at most tol.\n";
    		break;
    	    case 4:
    		s = "Fvec is orthogonal to the columns of the jacobian to machine precision.\n";
    		break;
    	    case 5:
    		s = "\n";
    		break;
    	    case 6:
    		s = "Tol is too small. No further reduction in the sum of squares is possible.\n";
    		break;
    	    case 7:
    		s = "Tol is too small. No further improvement in the approximate solution A is possible.\n";
    		break;
    	    }
    	    stufftext(s, 0);
    	    stufftext("\n", 0);
    	}
    } /* endif (nsteps) */
    
    if (!nsteps || XmToggleButtonGetState(nonl_autol_item)) {
    	/* check if the set is already allocated */
    	if ((nlloadset == -1) || (nlloadset == nlsetno) || !getsetlength(cg, nlloadset)) {
    	    nlloadset = nextset(cg);
    	    if (nlloadset == -1) {
    	      errwin("No more sets!");
    	      return;
    	    } else {
    		activateset(cg, nlloadset);
    		setlength(cg, nlloadset, 1);
    		setcomment(cg, nlloadset, fstr);
    	    }
    	}
    	
    	
    	
    	loadto = (int) GetChoice(nonl_load_item);
    	switch (loadto) {
    	case 0:
    	  sprintf(buf, "Evaluating fitted values and loading result to set %d:\n", nlloadset);
    	  stufftext(buf, 0);
    	  npts = getsetlength(cg, nlsetno);
    	  setlength(cg, nlloadset, npts);
    	  copycol2(cg, nlsetno, cg, nlloadset, 0);
    	  break;
    	case 1:
    	  sprintf(buf, "Evaluating fitted values and loading residuals to set %d:\n", nlloadset);
    	  stufftext(buf, 0);
    	  npts = getsetlength(cg, nlsetno);
    	  setlength(cg, nlloadset, npts);
    	  copycol2(cg, nlsetno, cg, nlloadset, 0);
    	  break;
    	case 2:
    	  sprintf(buf, "Computing fitting function and loading result to set %d:\n", nlloadset);
    	  stufftext(buf, 0);
    	  
    	  npts  = atoi((char *) xv_getstr(nonl_npts_item));
    	  if (npts <= 1) {
    	      errmsg("Number of points must be > 1");
    	      unset_wait_cursor();
    	      return;
    	  }
    	  start = atof((char *) xv_getstr(nonl_start_item));
    	  stop  = atof((char *) xv_getstr(nonl_stop_item));
    	  
    	  setlength(cg, nlloadset, npts);
    	  
    	  delx = (stop - start)/(npts - 1);
    	  xtmp = getx(cg, nlloadset);
	  for (i = 0; i < npts; i++) {
	      xtmp[i] = start + i * delx;
	  }
    	  break;
    	}
    	
    	do_compute(nlloadset, 0, cg, fstr);
    	
    	if (loadto == 1) { /* load residuals */
    	    ytmp = gety(cg, nlloadset);
    	    for (i = 0; i < npts; i++) {
	      ytmp[i] -= y[i];
	    }
    	}
    	
    	update_set_lists(cg);
    	drawgraph();
    }
    unset_wait_cursor();
}

static void destroy_nonl_frame(Widget w, XtPointer client_data, XtPointer call_data)
{
    int value = (int) client_data;
    
    if (value == NONL_CANCEL) {
    	killset(cg, nlloadset);
    	update_all(cg);
    	drawgraph();
    }
    
    nlloadset = -1;
    XtUnmanageChild(nonl_frame);
}

