;; Periodic table for NanoCAD project. 
;; 
;; Revision 1 
;;
;; Copyright 1997 Michael Deutschmann
;;
;; This software is subject to the GNU General Public License, version 2 
;; or later (your choice), as published by the Free Software Foundation.
;;
;; THERE ARE NO WARRANTIES WHATSOEVER.

;; Functions exported
;; 
;; (element-symbol <atomic number>) -> string 
;;   Get two-letter symbol for this element. 
;; (element-name <atomic number>) -> string
;;   Get name of this element.
;; (element-period <atomic number) -> number
;;   Find period of this element
;; (element-group <atomic number>) -> number
;;   Return number of main group (1-8), or #f if element is transistion.
;; (element-deviation <atomic number>) -> symbol
;;   Return the nature of any deviation from "aufbau" electron filling: 
;;   < Not useful now, but may be of use to any future hacker trying to 
;;     tackle metal complexes. >
;;
;;     #f           - No deviation
;;     'one-s-to-d  - One electron is moved from Ns to (N-1)d
;;     'two-s-to-d  - Both electrons are moved from Ns to (N-1)d
;;     'one-f-to-d  - One electron is moved from (N-2)f to (N-1)d
;;     'two-f-to-d  - Two electrons are moved from (N-2)f to (N-1)d
;;
;;

(define element-symbol '())
(define element-name '())

(let
 ((element-names
   #(
     ("H" "hydrogen")
     ("He" "helium")
   
     ("Li" "lithium")
     ("Be" "beryllium")
     ("B" "boron")
     ("C" "carbon")
     ("N" "nitrogen")
     ("O" "oxygen")
     ("F" "flourine")
     ("Ne" "neon")

     ("Na" "sodium")
     ("Mg" "magnesium")
     ("Al" "aluminum")
     ("Si" "silicon")
     ("P" "phosphorus")
     ("S" "sulfur")
     ("Cl" "chlorine")
     ("Ar" "argon")

     ("K" "potassium")
     ("Ca" "calcium")
     ("Sc" "scandium")
     ("Ti" "titanium")
     ("V" "vanadium")
     ("Cr" "chromium")
     ("Mn" "manganese")
     ("Fe" "iron")
     ("Co" "cobalt")
     ("Ni" "nickel")
     ("Cu" "copper")
     ("Zn" "zinc")
     ("Ga" "gallium")
     ("Ge" "germanium")
     ("As" "arsenic")
     ("Se" "selenium")
     ("Br" "bromine")
     ("Ar" "argon")

     ("Rb" "rubidium")
     ("Sr" "strontium")
     ("Y" "yttrium")
     ("Zr" "zirconium")
     ("Nb" "niobium")
     ("Mo" "molybdenum")
     ("Tc" "technetium")
     ("Ru" "rubidium")
     ("Rh" "rhodium")
     ("Pd" "palladium")
     ("Ag" "silver")
     ("Cd" "cadmium")
     ("In" "indium")
     ("Sn" "tin")
     ("Sb" "antimony")
     ("Te" "tellurium") 
     ("I" "iodine")
     ("Xe" "xenon")
   
     ("Cs" "cesium")
     ("Ba" "barium")
     ("La" "lanthanum")
     ("Ce" "cerium")
     ("Pr" "praseodymium")
     ("Nd" "neodymium")
     ("Pm" "promethium")
     ("Sm" "samarium")
     ("Eu" "europium")
     ("Gd" "gadolinium")
     ("Tb" "terbium")
     ("Dy" "dysprosium")
     ("Ho" "holmium")
     ("Er" "erbium")
     ("Tm" "thulium") 
     ("Yb" "ytterbium")
     ("Lu" "lutetium")
     ("Hf" "hafnium")
     ("Ta" "tantalum")
     ("W" "tungsten")
     ("Re" "rhenium")
     ("Os" "osmium")
     ("Ir" "iridium")
     ("Pt" "platinum")
     ("Au" "gold")
     ("Hg" "mercury")
     ("Tl" "thallium")
     ("Pb" "lead")
     ("Bi" "bismuth")
     ("Po" "polonium")
     ("At" "astatine")
     ("Rn" "radon")

     ("Fr" "francium")
     ("Ra" "radium")
     ("Ac" "actinium")
     ("Th" "thorium")
     ("Pa" "protactinium")
     ("U" "uranium")
     ("Np" "neptunium")
     ("Pu" "plutonium")
     ("Am" "americium")
     ("Cm" "curium")
     ("Bk" "berkelium")
     ("Cf" "californium")
     ("Es" "einsteinium")
     ("Fm" "fermium")
     ("Md" "mendelevium")
     ("No" "nobelium")   
     ("Lr" "lawrencium")
     ("Db" "dubnium") 
     ("Jl" "joliotium")
     ("Rf" "rutherfordium")
     ("Bh" "bohrium")
     ("Hn" "hahnium")
     ("Mt" "meitnerium")
 )) )
 (set! element-symbol 
  (lambda (number) (car (vector-ref element-names (- number 1))))
 )
 (set! element-name 
  (lambda (number) (cadr (vector-ref element-names (- number 1))))
))

(define element-period
 (lambda (number)
  (cond
   ((< number 3) 1)
   ((< number 11) 2)
   ((< number 19) 3)
   ((< number 37) 4)
   ((< number 55) 5)
   ((< number 87) 6)
   (else 7)
)))

(define element-group
 (lambda (number)
  (cond
   ((= number 1) 1)
   ((= number 2) 8)
   ((< number 11) (- number 2))
   ((< number 19) (- number 10))
   ((< number 21) (- number 18))
   ((< number 31) #f)
   ((< number 37) (- number 28))
   ((< number 39) (- number 36))
   ((< number 49) #f)
   ((< number 55) (- number 46))
   ((< number 57) (- number 54))
   ((< number 81) #f)
   ((< number 87) (- number 78))
   ((< number 89) (- number 86))
   (else #f)
))) 
 
(define element-deviation
 (lambda (number)
  (case number
   ((24 29 41 42 43 44 45 47 78 79) 'one-s-to-d)
   ((46) 'two-s-to-d)
   ((57 58 64 89 91 92 93 96) 'one-f-to-d)
   ((90) 'two-f-to-d)
   (else #f)
)))
