;; Hybridization module for NanoCAD project. 
;; 
;; Revision 1 
;;
;; Copyright 1997 Michael Deutschmann
;;
;; This software is subject to the GNU General Public License, version 2 
;; or later (your choice), as published by the Free Software Foundation.
;;
;; THERE ARE NO WARRANTIES WHATSOEVER.
 
;; Find hybridization of an atom at a given valency.
;;
;; (hybridization <atomic-number> <charge> <bonds> <valen>)
;; 
;; <atomic-number> Atomic number of atom concerned
;; <charge> Charge of the ion
;; <bonds> Number of bonds attached (double/triple bonds count once)
;; <valen> Valency                  (double/triple bonds count twice/thrice) 
;;
;; Returns:
;;  #f       - this valency not possible for this atom/ion
;;
;; or a pair of hybridization and number of lone pairs in cdr
;;
;;  'nothing - ion has no electrons (H+)
;;  
;;  's       - unhybridized s orbital. (H_2)
;;  'p       - unhybridized p orbital(s). (HF)
;;             (note that if this is selected, any lone pair
;;              occupying the s orbital is not counted.)
;;
;;  'sp      - sp (linear) hybridization (BeCl_2)
;;  'sp2     - sp2 (trigonal planar) hybridization (BCl3)
;;  'sp3     - sp3 (tetrahedral) hybridization (CH_4)
;;  'sp3d    - sp3d (trigonal bipyramid) hybridization (SF_4)
;;  'sp3d2   - sp3d2 (octahedral) hybridization (SF_6)
;;
;;  The following two are not recognized, to my knowledge, but I added them
;;  to cover all possible cases of noble gas hypervalence.
;;  'sp3d3   - for xenon hexaflouride 
;;  'sp3d4   - for a hypothetical AX8 molecule 
;;             (such has never been observed)
;;  
;; Errata:
;; * Transistion metals and ions are not supported at all. (I'm not sure 
;;   how to express complexes in this framework. The more I read my
;;   inorganic chemistry textbook, the harder the problem seems.)
;;
;; * The code assumes that all ions bond identically to the atom of same 
;;   electron number, which is generally only true of main group ions. 
;;   However due to the previous problem this is a small detail.
;;
;; * There is evidence that some species that my algorithm assigns an sp3 
;;   hybridization actually use unhybridized p orbitals. (ie: hydrogen 
;;   sulphide is closer to 90 degrees than the 109.5 expected). I haven't 
;;   installed this case, since I don't know how to determine when it 
;;   applies.
;;
;; * I included handling for electron stripped species for completeness
;;   (Ie: H+, He++, Li+++), although they won't be very useful at the 
;;   moment as they can't bond. Same for Helium and Neon (and isoelectronic
;;   species).
;;
(define hybridization
 (let ((normal-hybrids #(s sp sp2 sp3 sp3d sp3d2 sp3d3 sp3d4)))
  (lambda (atomic-no charge bonds valence)
   (let* 
    ((electron-no (+ atomic-no charge)) 
     (group (element-group electron-no))
    )
    (cond
     ; Sorry - transistion metal bonding not implemented.
     ((not group) #f)

     ; duet rule cases
     ((< electron-no 3)
      (case electron-no
       ((0) (if (zero? valence) '(nothing . 0) #f))
       ((1) (if (= valence 1) '(s . 0) #f))
       ((2) (if (zero? valence) '(s . 1) #f))
     ))

     ; Special case for divalent carbon group compounds
     ; (singlet methylene, tin (II) chloride)
     ((and (= group 4) (= bonds 2) (= valence 2)) '(sp2 . 1))

     ; Electron-deficent or carbon-group case
     ((< group 5)
      (if (= valence group)
       (cons (vector-ref normal-hybrids (-1+ bonds)) 0)
       #f 
     ))

     ; Univalent halogens. Without this exception we would regard them
     ; as '(sp3 . 3). Although I see no problem with that, my textbook 
     ; shows them bonding with p orbitals. Presumably they have their 
     ; reasons.
     ((and (= group 7) (= valence 1)) '(p . 2))
    
     ; General non-electron-deficent case
     ((> group 4) 
      (let
       ((hyper-valence (- valence (- 8 group))))
       (if 
        (or
         (zero? hyper-valence)
         (and
          (even? hyper-valence)
          (> (element-period electron-no) 2)
        ))
        (let ((lone-pairs (- group 4 (/ hyper-valence 2)))) 
         (cons
          (vector-ref normal-hybrids (+ bonds lone-pairs -1))
          lone-pairs
        ))
        #f
     )))
     (else #f)
)))))
