/*
 * Copyright (c) 1997-1999 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    <http://java.apache.org/>."
 *
 * 4. The names "Apache JServ", "Apache JServ Servlet Engine" and 
 *    "Java Apache Project" must not be used to endorse or promote products 
 *    derived from this software without prior written permission.
 *
 * 5. Products derived from this software may not be called "Apache JServ"
 *    nor may "Apache" nor "Apache JServ" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    <http://java.apache.org/>."
 *    
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Group. For more information
 * on the Java Apache Project and the Apache JServ Servlet Engine project,
 * please see <http://java.apache.org/>.
 *
 */

package org.apache.java.recycle.pool;

import org.apache.java.lang.*;
import org.apache.java.recycle.*;

/**
 * This class extends the Thread class to add recyclable functionalities.
 *
 * @author <a href="mailto:stefano@apache.org">Stefano Mazzocchi</a>
 * @version $Revision: 1.4 $ $Date: 1999/10/08 22:13:21 $
 */

class Worker extends Thread implements Recyclable {

	private Stoppable work;
    private RecycleBin pool;
	private boolean alive;

    /**
     * Allocates a new <code>Worker</code> object.
     */
    protected Worker(ThreadGroup group, RecycleBin pool, String name) {
        super(group, name);
        this.pool = pool;
        this.work = null;
        this.alive = true;
        setDaemon(false);
    	
        if (WorkerPool.DEBUG) System.out.println(this.getName() + ": created.");
        if (WorkerPool.QUICK_DEBUG) System.out.write('#');
    }

	/**
	 * The main execution loop.
	 */
	public final synchronized void run() {
        if (WorkerPool.DEBUG) System.out.println(this.getName() + ": starting.");
        if (WorkerPool.QUICK_DEBUG) System.out.write('$');
		
        // Notify the pool this worker started running.
        this.notifyAll();

		while (alive) {
            while (work == null) {
                try {
                    if (WorkerPool.DEBUG) System.out.println(this.getName() + ": waiting.");
                    if (WorkerPool.QUICK_DEBUG) System.out.write('*');
                    wait();
                    if (WorkerPool.DEBUG) System.out.println(this.getName() + ": notified.");
                    if (WorkerPool.QUICK_DEBUG) System.out.write(']');
                } catch (InterruptedException ignored) {}
            }

			if (WorkerPool.DEBUG) System.out.println(this.getName() + ": running.");
            if (WorkerPool.QUICK_DEBUG) System.out.write('(');
			
			try {
				this.work.run();
			} catch (ThreadDeath death) {
				if (WorkerPool.DEBUG) System.out.println(this.getName() + ": thread has died.");
                if (WorkerPool.QUICK_DEBUG) System.out.write('!');
				
				// This is to let the thread death propagate to the runtime
				// enviroment to let it know it must kill this worker
				throw death;
			} catch (Throwable error) {
				// Error thrown while working.
				if (WorkerPool.DEBUG) System.out.println(this.getName() + ": error caught: " + error);
                if (WorkerPool.QUICK_DEBUG) System.out.write('?');
				// XXX: what should we do when this happens?
			}	

			if (WorkerPool.DEBUG) System.out.println(this.getName() + ": done.");
            if (WorkerPool.QUICK_DEBUG) System.out.write(')');
			
			// recycle ourselves
			this.pool.recycle(this);
		}
	}

	/**
	 * Set the <code>Work</code> code this <code>Worker</code> must
	 * execute and <i>notifies</i> its thread to do it.
	 */
	protected synchronized void execute(Stoppable work) {
		if (WorkerPool.DEBUG) System.out.println(this.getName() + ": notifying this worker.");
        if (WorkerPool.QUICK_DEBUG) System.out.write('[');

		this.work = work;
        notify();
	}

	/**
	 * Cleans this worker by setting its <code>Work</code> code to null.
	 */
	public void clean() {
		if (WorkerPool.DEBUG) System.out.println(this.getName() + ": cleaning.");
        if (WorkerPool.QUICK_DEBUG) System.out.write('0');
		
		this.work = null;
	}

	/**
	 * Set the <code>alive</code> variable to false causing the worker to die.
	 * If the worker is stalled and a timeout generated this call, this method
	 * does not change the state of the worker (that must be destroyed in other
	 * ways).
	 */
	public void destroy() {
		if (WorkerPool.DEBUG) System.out.println(this.getName() + ": destroying.");
        if (WorkerPool.QUICK_DEBUG) System.out.write('X');

		this.alive = false;
		this.work.stop();
	}
}
