/*
 * $Header: /home/cvs/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/deploy/SecurityCollection.java,v 1.1 2000/02/13 01:43:45 craigmcc Exp $
 * $Revision: 1.1 $
 * $Date: 2000/02/13 01:43:45 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 


package org.apache.tomcat.deploy;


import java.util.Enumeration;
import java.util.Hashtable;


/**
 * Representation of a web resource collection for a web application's security
 * constraint, as represented in a <code>&lt;web-resource-collection&gt;</code>
 * element in the deployment descriptor.
 * <p>
 * <b>WARNING</b>:  The property setting methods in this class are for use by
 * the application logic that parses a web application deployment descriptor.
 * They should not be called by a Context implementation (or an associated
 * Valve or Interceptor that implements the authentication and authorization
 * constraints expressed here).
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.1 $ $Date: 2000/02/13 01:43:45 $
 */

public final class SecurityCollection {


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a new security collection instance with default values.
     *
     * @param constraint Security constraint we are attached to
     * @param name Name of this security collection
     */
    public SecurityCollection(SecurityConstraint constraint, String name) {

	super();
	this.constraint = constraint;
	this.name = name;

    }


    // ----------------------------------------------------- Instance Variables


    /**
     * The security constraint we are attached to.
     */
    private SecurityConstraint constraint = null;


    /**
     * The HTTP methods covered by this web resource collection.
     */
    private String[] methods = new String[0];


    /**
     * The name of this web resource collection.
     */
    private String name = null;


    /**
     * The URL patterns protected by this security collection, keyed by
     * pattern.
     */
    private Hashtable patterns = new Hashtable();


    // ------------------------------------------------------------- Properties


    /**
     * Return the security constraint we are attached to.
     */
    public SecurityConstraint getConstraint() {

	return (this.constraint);

    }


    /**
     * Return the name of this web resource collection.
     */
    public String getName() {

	return (this.name);

    }


    // --------------------------------------------------------- Public Methods


    /**
     * Add an HTTP request method to be part of this web resource collection.
     */
    public void addMethod(String method) {

	if (method == null)
	    return;
	synchronized (methods) {
	    String results[] = new String[methods.length + 1];
	    for (int i = 0; i < methods.length; i++) {
		if (method.equals(methods[i]))
		    return;
		results[i] = methods[i];
	    }
	    results[results.length - 1] = method;
	    methods = results;
	}

    }


    /**
     * Add a URL pattern to be part of this web resource collection.
     */
    public void addPattern(String pattern) {

	patterns.put(pattern, pattern);
	constraint.addPattern(pattern, this);

    }


    /**
     * Return <code>true</code> if the specified HTTP request method is
     * part of this web resource collection.
     *
     * @param method Request method to check
     */
    public boolean findMethod(String method) {

	if (method == null)
	    return (false);
	if (methods.length == 0)
	    return (true);
	for (int i = 0; i < methods.length; i++) {
	    if (method.equals(methods[i]))
		return (true);
	}
	return (false);

    }


    /**
     * Is the specified pattern part of this web resource collection?
     *
     * @param pattern Pattern to be compared
     */
    public boolean findPattern(String pattern) {

	return (patterns.get(pattern) != null);

    }


    /**
     * Return the set of URL patterns that are part of this web resource
     * collection.  If none have been specified, a zero-length array is
     * returned.
     */
    public String[] findPatterns() {

	synchronized (patterns) {
	    String results[] = new String[patterns.size()];
	    Enumeration urls = patterns.keys();
	    for (int i = 0; i < results.length; i++)
		results[i] = (String) urls.nextElement();
	    return (results);
	}

    }


    /**
     * Remove the specified HTTP request method from those that are part
     * of this web resource collection.
     *
     * @param method Request method to be removed
     */
    public void removeMethod(String method) {

	if (method == null)
	    return;
	synchronized (methods) {
	    int n = -1;
	    for (int i = 0; i < methods.length; i++) {
		if (method.equals(methods[i])) {
		    n = i;
		    break;
		}
	    }
	    if (n < 0)
		return;
	    String results[] = new String[methods.length - 1];
	    int j = 0;
	    for (int i = 0; i < methods.length; i++) {
		if (i == n)
		    continue;
		results[j++] = methods[i];
	    }
	    methods = results;
	}

    }


    /**
     * Remove the specified URL pattern from those that are part of this
     * web resource collection.
     *
     * @param pattern Pattern to be removed
     */
    public void removePattern(String pattern) {

	patterns.remove(pattern);
	constraint.removePattern(pattern);

    }


}
