/*
 * $Header: /home/cvs/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/valves/InterceptorValve.java,v 1.1 2000/01/31 04:38:46 craigmcc Exp $
 * $Revision: 1.1 $
 * $Date: 2000/01/31 04:38:46 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 


package org.apache.tomcat.valves;


import java.io.IOException;
import java.util.Enumeration;
import java.util.Vector;
import javax.servlet.ServletException;
import org.apache.tomcat.Container;
import org.apache.tomcat.Interceptor;
import org.apache.tomcat.Lifecycle;
import org.apache.tomcat.LifecycleException;
import org.apache.tomcat.Request;
import org.apache.tomcat.Response;
import org.apache.tomcat.Valve;
import org.apache.tomcat.util.StringManager;


/**
 * Implementation of a <b>Valve</b> that supports a arbitrary sized stack of
 * <b>Interceptors</b> that are given the opportunity to pre-process and
 * post-process the current request and response.  When <code>invoke()</code>
 * is called, the following actions take place:
 * <ul>
 * <li>The <code>preService()</code> method of each configured Interceptor
 *     is called, until one of them throws an exception or returns
 *     <code>false</code> (indicating that the response has been generated).
 *     The Interceptors are called in the order they were added to this Valve.
 * <li>If no exception was thrown, and no Interceptor returned
 *     <code>false</code>, the next Valve in our pipeline is invoked.
 * <li>For each Interceptor whose <code>preService()</code> was called,
 *     the corresponding <code>postService()</code> method is called.  The
 *     Interceptors are called in reverse of the order they were added to
 *     this Valve.
 * <ul>
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.1 $ $Date: 2000/01/31 04:38:46 $
 */

public final class InterceptorValve
    extends ValveBase
    implements Lifecycle {


    // ----------------------------------------------------- Instance Variables


    /**
     * The descriptive information related to this implementation.
     */
    private static final String info =
	"org.apache.tomcat.core.InterceptorValve/1.0";


    /**
     * The set of Interceptors configured for this Valve.
     */
    private Interceptor interceptors[] = new Interceptor[0];


    /**
     * The StringManager for this package.
     */
    private StringManager sm =
	StringManager.getManager(Constants.Package);


    /**
     * Has this component been started yet?
     */
    private boolean started = false;


    // ------------------------------------------------------------- Properties


    /**
     * Return descriptive information about this Valve implementation.
     */
    public String getInfo() {

	return (info);

    }


    //---------------------------------------------------------- Public Methods


    /**
     * Add a new Interceptor to those associated with this Valve.  The
     * <code>preService()</code> method of this Interceptor will be the
     * last one called prior to processing the request, and its
     * <code>postService()</code> method will be the last one called
     * afterwards.
     *
     * @param interceptor New interceptor to be associated
     *
     * @exception IllegalArgumentException if this Interceptor refuses
     *  to become associated with our associated Container
     */
    public void addInterceptor(Interceptor interceptor) {

	synchronized (interceptors) {
	    interceptor.setContainer(container);	// May throw IAE
	    Interceptor temp[] = new Interceptor[interceptors.length + 1];
	    for (int i = 0; i < interceptors.length; i++)
		temp[i] = interceptors[i];
	    temp[interceptors.length] = interceptor;
	    interceptors = temp;
	}

    }


    /**
     * Return the set of Interceptors associated with this Valve.
     */
    public Interceptor[] findInterceptors() {

	synchronized (interceptors) {
	    Interceptor results[] = new Interceptor[interceptors.length];
	    for (int i = 0; i < interceptors.length; i++)
		results[i] = interceptors[i];
	    return (results);
	}

    }


    /**
     * Invoke the <code>preService()</code> of all associated Interceptors,
     * then invoke the next Valve in our pipeline, and finally call the
     * <code>postService()</code> method of all associated Interceptors.
     * This processing schedule will be interrupted if an Interceptor
     * returns <code>false</code> (indicating that it has completely
     * generated the response), or if it throws an exception.  In all cases,
     * the <code>postService()</code> method is guaranteed to be called for
     * every Interceptor whose <code>preService()</code> method was called
     * for this request.
     *
     * @param request The servlet request to be processed
     * @param response The servlet response to be created
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet error occurs
     */
    public void invoke(Request request, Response response)
	throws IOException, ServletException {

	// Acquire a private copy of the current Interceptor list
	// IMPLEMENTATION NOTE:  Not required if configuration can only
	// occur at startup time
	Interceptor list[] = findInterceptors();

	// Call the preService() method of all defined Interceptors
	boolean done = false;
	int last = -1;
	Throwable t = null;
	for (int i = 0; i < list.length; i++) {
	    last = i;
	    try {
		if (!list[i].preService(request, response))
		    done = true;
	    } catch (Throwable e) {
		if ((e instanceof IOException) ||
		    (e instanceof ServletException))
		    t = e;
		done = true;
	    }
	    if (done)
		break;
	}

	// Invoke the next Valve in our pipeline, if any
	if (getNext() != null) {
	    try {
		getNext().invoke(request, response);
	    } catch (Throwable e) {
		if ((e instanceof IOException) ||
		    (e instanceof ServletException))
		    t = e;
	    }
	}

	// Call the postService() methods of all Interceptors for which
	// we called the preService method
	if (last < 0)
	    return;
	for (int i = last; i >= 0; i--) {
	    list[i].postService(request, response, t);
	}

    }


    /**
     * Remove this Interceptor from those associated with this Valve.  If
     * no such association exists, no action is taken.
     *
     * @param interceptor Interceptor to be removed
     */
    public void removeInterceptor(Interceptor interceptor) {

	synchronized (interceptors) {
	    boolean found = false;
	    Interceptor temp[] = new Interceptor[interceptors.length - 1];
	    for (int i = 0; i < interceptors.length; i++) {
		if (!found) {
		    if (interceptors[i] == interceptor) {
			found = true;
			try {
			    interceptor.setContainer(null);
			} catch (IllegalArgumentException e) {
			    ;
			}
			continue;
		    }
		}
		if (!found)
		    temp[i] = interceptors[i];
		else
		    temp[i-1] = interceptors[i];
	    }
	    if (found)
		interceptors = temp;
	}

    }


    // ------------------------------------------------------ Lifecycle Methods


    /**
     * Prepare for the beginning of active use of the public methods of this
     * component.  This method should be called after <code>configure()</code>,
     * and before any of the public methods of the component are utilized.
     *
     * @exception IllegalStateException if this component has already been
     *  started
     * @exception LifecycleException if this component detects a fatal error
     *  that prevents this component from being used
     */
    public void start() throws LifecycleException {

	// Validate and update our current component state
	if (started)
	    throw new LifecycleException
		(sm.getString("interceptorValve.alreadyStarted"));
	started = true;

	// Start our subordinate Interceptors
	synchronized (interceptors) {
	    for (int i = 0; i < interceptors.length; i++) {
		if (interceptors[i] instanceof Lifecycle)
		    ((Lifecycle) interceptors[i]).start();
	    }
	}

    }


    /**
     * Gracefully terminate the active use of the public methods of this
     * component.  This method should be the last one called on a given
     * instance of this component.
     *
     * @exception IllegalStateException if this component has not been started
     * @exception LifecycleException if this component detects a fatal error
     *  that needs to be reported
     */
    public void stop() throws LifecycleException {

	// Validate and update our current state
	if (!started)
	    throw new LifecycleException
		(sm.getString("interceptorValve.notStarted"));
	started = false;

	// Stop our subordinate Interceptors
	synchronized (interceptors) {
	    for (int i = interceptors.length - 1; i >= 0; i--) {
		if (interceptors[i] instanceof Lifecycle)
		    ((Lifecycle) interceptors[i]).stop();
	    }
	}

    }


}
