/*****
  This file is part of the Babel Program
  Copyright (C) 1992-94 W. Patrick Walters and Matthew T. Stahl
  The Babel Program is a product of the Dolata Research Group
  Dept. of Chemistry
  University of Arizona
  Tucson, AZ 85721
  
  For more information please contact :
  
  babel@mercury.aichem.arizona.edu
  ------------------------------------------------------------------------
  FILE : convert.c
  AUTHOR(S) : Pat Walters
  DATE : 4-93
  PURPOSE : main program
  ******/

#include "bbltyp.h"
#include "bblmast.h"

static warning wstr;

static char *program_name;
int use_title = FALSE;

#define UPPER_LIMIT 1000000

#ifndef LIBRARY
int main(int argc, char *argv[])
{
  ums_type *mol = NULL;
  int need_to_do_outputs = FALSE;
  
  babel_init();
  program_name = argv[0];
  if (argc == 1)
  {
    usage();
    exit(0);
  }    
  
  mol = (ums_type *)malloc(sizeof(ums_type));
  mol->control = (bbl_control *)malloc(sizeof(bbl_control));
  if (mol == NULL)
    fatal_error("Could not allocate memory for UMS");
  
  init_babel_control(mol);
  Size = MASTERSIZE;
  
  process_command_line_args(argc,argv,mol);
  if ((OutfileType == none) && (UseMenus == FALSE))
    fatal_error("No output flag specified");
  
  if (Spline)
  {
    do_spline(mol);
    return(0);
  }
  
  if (UseMenus)
    babel_menus(mol);
  else
  {
    need_to_do_outputs = do_inputs(mol);
    if (need_to_do_outputs == TRUE)
      do_outputs(mol);
  }
  
  return(0);
}
#endif

void new_control(ums_type *mol)
{

  mol->control = (bbl_control *)malloc(sizeof(bbl_control));

  if (!mol->control)
    fatal_error("Unable to allocate memory for babel control");
  
  init_babel_control(mol);
}


void init_babel_control(ums_type *mol)
{
  strcpy(InfileName,"");
  strcpy(OutfileName,"");
  strcpy(InputKeywords,NOKEY);
  strcpy(OutputKeywords,NOKEY);    
  strcpy(DeleteStr,"");         
  InfileType = none;
  OutfileType = none;
  UseMenus = FALSE;
  Verbose = FALSE;
  AddHydrogens = FALSE;
  DeleteAtoms = FALSE;
  ReaderFunction = NULL;
  WriterFunction = NULL;
  LowerLimit = 1;
  UpperLimit = UPPER_LIMIT;
  Multi = single_struct;
  Spline = FALSE;
  Align = FALSE;
  CenterMol = FALSE;
  Increment = 0;
  DoRenum = FALSE;
}

void 
  process_command_line_args(int argc, char *argv[],ums_type *mol)
{
  char tempstr[50];
  
  while (argc > 1) 
  {
    if (argv[1][0] == '-')
    {
      switch (argv[1][1])
      {
      case 'r' :
	lowercase(&argv[1][1]);
	if EQ(&argv[1][1],"renum")
	  DoRenum = TRUE;
	if ((argv[2] != NULL) && (argv[2][0] != '-'))
	{
	  if (isdigit(argv[2][0]))
	    NewBase = atoi(argv[2]);
	  else
	    NewBase = 1;
	}
	else
	  NewBase = 1;
	break;
      case 'a' :
	lowercase(&argv[1][1]);
	if EQ(&argv[1][1],"align")
	  Align = TRUE;
	break;
      case 'b' :
	write_blurb();
	exit(0);
	break;
      case 'c' :
	lowercase(&argv[1][1]);
	if (EQ(&argv[1][1],"center"))
	  CenterMol = TRUE;
	break;
      case 'v':
	lowercase(&argv[1][1]);
	if (EQ(&argv[1][1],"version"))
	{
	  printf("Babel version %s\n",BABEL_VERSION);
	  exit(0);
	}
	else
	  Verbose = TRUE;
	break;
      case 'm':
	Verbose = TRUE;
	UseMenus = TRUE;
	break;
      case 'h':
	lowercase(&argv[1][1]);
	if (EQ(&argv[1][1],"help"))
	{
	  if (argv[2] == NULL)
	    usage();
	  else
	  {
	    if (EQ(argv[2],"inputs"))
	      show_inputs();
	    else
	      if (EQ(argv[2],"outputs"))
		show_outputs();
	  }
	  exit(0);
	}
	else
	AddHydrogens = TRUE;
	break;
      case 's':
	Spline = TRUE;
	if ((argv[2] != NULL) && (argv[2][0] != '-'))
	  Increment = atoi(argv[2]) + 2;
	break;
      case 'd':
	DeleteAtoms = TRUE;
	if ((argv[2] != NULL) && (argv[2][0] != '-'))
	  strcpy(DeleteStr,argv[2]);
	else 
	{
	  sprintf(wstr,"Default type to be deleted is hydrogen");
	  show_warning(wstr);
	  strcpy(DeleteStr,"default");
	}
	break;
      case 'i':
	translate_input_code(&argv[1][2],mol);
	if (InfileType == none)
	{
	  sprintf(wstr,"%s is not a supported input file type \n",argv[1]);
	  fatal_error(wstr);
	}
	strcpy(InfileName,argv[2]);
	if ((argv[3] != NULL) && (argv[3][0] != '-'))
	  strcpy(InputKeywords,argv[3]);
	else 
	  strcpy(InputKeywords,"KEYWORDS GO HERE");
	set_limits(mol);
	break;
      case 'o':
	translate_output_code(&argv[1][2],mol);
	if (OutfileType == none)
	{
	  sprintf(wstr,"%s is not a supported output file type \n",argv[1]);
	  fatal_error(wstr);
	}
	if (argv[2] != NULL)
	{
	  strcpy(OutfileName,argv[2]);
	  strcpy(BaseName,OutfileName);
	  strcpy(tempstr,OutfileName);
	  uppercase(tempstr);
	  if (EQ(gettoken(tempstr,".",1),"TITLE"))
	  {
	    Multi = multi_struct;
	    use_title = TRUE;
	    if (gettoken(OutfileName,".",2) != NULL)
	      strcpy(DefaultExtension,gettoken(OutfileName,".",2));
	    else
	      DefaultExtension[0] = '\0';
	  }
	  else
	    Multi = multi_conf;
	}
	else
	  strcpy(OutfileName,"CON");
	if ((argv[3] != NULL) && (argv[3][0] != '-'))
	  strcpy(OutputKeywords,argv[3]);
	else 
	  strcpy(OutputKeywords,"KEYWORDS GO HERE");
      }
    }
    argv++;
    argc--;
  }
}



void translate_input_code(char *code, ums_type *mol)
{
  int i;
  
  for (i = 0; i < MASTERSIZE; i++)
    if (EQ(code,master[i].code) && (master[i].operation == input))
    {
      InputInfo = master[i];
    }
}

void translate_output_code(char *code,ums_type *mol)
{
  int i;
  for (i = 0; i < MASTERSIZE; i++)
    if (EQ(code,master[i].code) && (master[i].operation == output))
    {
      OutputInfo = master[i];
    }
}

int do_inputs(ums_type *mol)
{		
  FILE *file1;
  int end = FALSE, result = 0;
  int in_count = 1;
  int out_count = 1;
  
  if (Verbose)
  {
    sprintf(wstr,"Reading %s file %s",InputTypeName,InfileName);
    show_warning(wstr);
  }
  file1 = open_read(InfileName);
  
  if (EQ(InputKeywords,"LAST"))
    Multi = single_struct;
	
  while (!end)
  {
    if (!end)
    {
      result = ReaderFunction(file1,mol);

      if (check_for_eof(file1))
	end = TRUE;
      if ((!end) && (EQ(InputKeywords,NOKEY)))
      {
	usage_multi_struct();
	exit(0);
      }

      if (in_count > UpperLimit)
	exit(0);
      
      if ((want_this_file(mol,in_count,end)) && result)
      {
	if (Multi != single_struct)
	  generate_outfile_name(mol,out_count);

	if (!UseMenus)
	{
	  do_outputs(mol);
	  out_count++;
	}
      }
      in_count++;
    }
  }
  return(FALSE);
}

void generate_outfile_name(ums_type *mol,int count)
{
  if (NOTEQ(OutfileName,"CON"))
  {
    if (Multi == multi_struct)
    {
      if (use_title)
      {
	if (DefaultExtension[0] != '\0')
	  sprintf(OutfileName,"%s.%s",Title,DefaultExtension);
	else
	  sprintf(OutfileName,"%s",Title);
      }
    }
    else
      if ((UpperLimit - LowerLimit) > 0)
      	make_output_name(BaseName,OutfileName,count);
    lowercase(OutfileName);
  }
}

void set_limits(ums_type *mol)
{
  char token1[25],token2[25];
  
  uppercase(InputKeywords);
  if (EQ(InputKeywords,"ALL"))
  {
    LowerLimit = 1;
    UpperLimit = 999999;
  }
  else
    if (strchr(InputKeywords,'-'))
    {
      strcpy(token1,gettoken(InputKeywords,"-",1));
      strcpy(token2,gettoken(InputKeywords,"-",2));
      
      if ((isdigit(token1[0])) && (isdigit(token2[0])))
      {
	LowerLimit = atoi(token1);
	UpperLimit = atoi(token2);
      }
    }
}


int want_this_file(ums_type *mol, int counter, int end)
{
  
  if ((counter == 1) && (end))
  {
    Multi = single_struct;
    return(TRUE);
  }

  if (EQ(InputKeywords,"LAST"))
    if (end)
    {
      return(TRUE);
    }
    else
      return(FALSE);

  switch(Multi)
  {
  case multi_conf :
    if ((counter >= LowerLimit) && (counter <= UpperLimit))
      return(TRUE);
    break;
  case multi_struct :  
    if (EQ(InputKeywords,"ALL"))
      return(TRUE);
    if ((strlen(Title) > 0) && (strstr(InputKeywords,Title) != NULL))
      return(TRUE);
    break;
    default :
      return(FALSE);
  }
  return(FALSE);
}



int do_outputs(ums_type *mol)
{		
  int result = 0;
  FILE *file1 = NULL;
  
  if (AddHydrogens)
    add_hydrogens(mol);
  if (DoRenum)
    mol = renum_for_zmat(mol,NewBase);
  if (CenterMol)
    center_at_origin(mol);
  if (Align)
    AlignMol(mol);
  if (DeleteAtoms)
    mol = delete_atoms(mol,DeleteStr);
  
  if (Atoms > 0)
  {  
    file1 = open_write(OutfileName);

    if (Title[0] == '\0')
      strcpy(Title,OutfileName);

    if (Verbose)
    {
      sprintf(wstr,"Writing %d Atoms and %d Bonds to %s file %s",
	      Atoms,Bonds,OutfileTypeName,OutfileName); 
      show_warning(wstr);
    }
    result = WriterFunction(file1,mol);
  }
  else
  {
    show_warning("No atoms found in this structure");
  }
  release_ums(mol);
  close_file(OutfileName,file1);
  return(result);
}


void 
  usage()
{
  int i;
  sprintf(wstr,"Babel %s",BABEL_VERSION);
  puts(wstr);
  sprintf(wstr,"for menus type -- %s -m\n",program_name);
  puts(wstr);
  sprintf(wstr,"Usage is : \n%s [-v] -i<input-type> <name> -o<output-type> <name> \"<keywords>\"\n",program_name);
  puts(wstr);
  sprintf(wstr,"\nCurrently supported input types\n");
  puts(wstr);
  show_inputs();
  sprintf(wstr,"\nCurrently supported output types\n");
  puts(wstr);
  show_outputs();
}  

void show_inputs()
{
  int i;
  
  for (i = 0; i < MASTERSIZE; i++)
    if (master[i].operation == input)
    {
      sprintf(wstr,"\t%s -- %s file",master[i].code,master[i].type_name);
      puts(wstr);
    }
}

void show_outputs()
{
  int i;
  
  for (i = 0; i < MASTERSIZE; i++)
    if (master[i].operation == output)
    {
      sprintf(wstr,"\t%s -- %s file",master[i].code,master[i].type_name);
      puts(wstr);
    }
}

int output_all_formats(char *file_name,ums_type *mol)
{
  int i;
  int result;
  char new_name[100];
  
  for (i = 0; i < MASTERSIZE; i++)
    if ((master[i].operation == output) && (master[i].type != diagnostics))
    {
      strcpy(new_name,file_name);
      strcat(new_name,".");
      strcat(new_name,master[i].code);
      strcpy(OutputKeywords,"KEYWORDS GO HERE");
      {
	sprintf(wstr,"Writing %s file %s",master[i].type_name,new_name); 
	show_warning(wstr);
      }
      result = master[i].func(new_name,mol);
    }
  return(TRUE);
}


void write_blurb()
{
  int i,j;
  
  printf("Babel will read the following file types :\n");  
  j = 0;
  for (i = 0; i < MASTERSIZE; i++)
    if (master[i].operation == input)
    {
      j++;
      printf("%-25s",master[i].type_name);
      if (((j % 3) == 0) && j > 0)
	printf("\n");
    }
  printf("\n");
  
  printf("\nBabel will write the following file types :\n");
  j = 0;
  for (i = 0; i < MASTERSIZE; i++)
    if ((master[i].operation == output) && (master[i].type != diagnostics))  
    {
      j++;
      printf("%-25s",master[i].type_name);
      if (((j % 3) == 0) && j > 0)
	printf("\n");
    }
  printf("\n");
}

int do_spline(ums_type *mol)
{		
  FILE *file1;
  int end = FALSE, result = 0;
  int i;
  int in_count = 0;
  int out_count = 1;
  ums_type *mol2,*tmp;
  vect_type **vect;
  
  mol2 = (ums_type *)malloc(sizeof(ums_type));
  if (!mol2)
    fatal_error("Unable to allocate memory for ums");
  mol2->control = mol->control;
  
  file1 = open_read(InfileName);
  
  vect = NULL;
  while (!end)
  {
    if (in_count == 0)
    {
      result = ReaderFunction(file1,mol);
      result = ReaderFunction(file1,mol2);
      in_count = 2;
      vect = (vect_type **)malloc(sizeof(vect_type *) * (Atoms+1));
      for (i = 1;i <= Atoms;i++)
	vect[i] = (vect_type *)malloc(sizeof(vect_type));
    }
    else
    {
      tmp = mol;
      mol = mol2;
      mol2 = tmp;
      release_ums(mol2);
      result = ReaderFunction(file1,mol2);
      in_count++;
    }
    
    get_vectors(mol,mol2,vect,Increment);

    for (i = 0;i < Increment;i++)
    {
      generate_outfile_name(mol,out_count);
      do_outputs(mol);
      out_count++;
      add_step(mol,vect);
    }

    if (check_for_eof(file1))
      end = TRUE;
  }

  return(FALSE);
}



