/*
 * $Header: /home/cvs/jakarta-tomcat-4.0/catalina/src/share/org/apache/catalina/authenticator/FormAuthenticator.java,v 1.7 2001/03/14 02:17:20 craigmcc Exp $
 * $Revision: 1.7 $
 * $Date: 2001/03/14 02:17:20 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */


package org.apache.catalina.authenticator;


import java.io.IOException;
import java.security.Principal;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.catalina.HttpRequest;
import org.apache.catalina.HttpResponse;
import org.apache.catalina.Realm;
import org.apache.catalina.Session;
import org.apache.catalina.deploy.LoginConfig;



/**
 * An <b>Authenticator</b> and <b>Valve</b> implementation of FORM BASED
 * Authentication, as described in the Servlet API Specification, Version 2.2.
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.7 $ $Date: 2001/03/14 02:17:20 $
 */

public final class FormAuthenticator
    extends AuthenticatorBase {


    // ----------------------------------------------------- Instance Variables


    /**
     * Descriptive information about this implementation.
     */
    private static final String info =
	"org.apache.catalina.authenticator.FormAuthenticator/1.0";


    // ------------------------------------------------------------- Properties


    /**
     * Return descriptive information about this Valve implementation.
     */
    public String getInfo() {

	return (this.info);

    }


    // --------------------------------------------------------- Public Methods


    /**
     * Authenticate the user making this request, based on the specified
     * login configuration.  Return <code>true</code> if any specified
     * constraint has been satisfied, or <code>false</code> if we have
     * created a response challenge already.
     *
     * @param request Request we are processing
     * @param response Response we are creating
     * @param login Login configuration describing how authentication
     *              should be performed
     *
     * @exception IOException if an input/output error occurs
     */
    public boolean authenticate(HttpRequest request,
				HttpResponse response,
				LoginConfig config)
	throws IOException {

	// Have we already authenticated someone?
	Principal principal =
	    ((HttpServletRequest) request.getRequest()).getUserPrincipal();
	if (principal != null) {
            if (debug >= 1)
                log("Already authenticated '" +
                    principal.getName() + "'");
	    return (true);
        }

	// Acquire references to objects we will need to evaluate
	HttpServletRequest hreq =
	    (HttpServletRequest) request.getRequest();
	HttpServletResponse hres =
	    (HttpServletResponse) response.getResponse();
	String contextPath = hreq.getContextPath();
	String requestURI = hreq.getRequestURI();
	response.setContext(request.getContext());
        Session session = null;

	// Is this a request for the login page itself?  Test here to avoid
	// displaying it twice (from the user's perspective) -- once because
	// of the "save and redirect" and once because of the "restore and
	// redirect" performed below.
	if (requestURI.equals(contextPath + config.getLoginPage())) {
            if (debug >= 1)
                log("Requesting login page normally");
	    return (true);	// Display the login page in the usual manner
        }

	// Is this the action request from the login page?
	boolean loginAction =
	    requestURI.startsWith(contextPath) &&
	    requestURI.endsWith(Constants.FORM_ACTION);

	// No -- Save this request and redirect to the form login page
	if (!loginAction) {
	    session = getSession(request, true);
            if (debug >= 1)
                log("Save request in session '" + session.getId() + "'");
	    saveRequest(request, session);
	    request.setRequestURI(contextPath + config.getLoginPage());
	    return (true);	// Display the login page in the usual manner
	}

	// Yes -- Validate the specified credentials and redirect
	// to the error page if they are not correct
	Realm realm = context.getRealm();
	String username = hreq.getParameter(Constants.FORM_USERNAME);
	String password = hreq.getParameter(Constants.FORM_PASSWORD);
	principal = realm.authenticate(username, password);
	if (principal == null) {
            if (debug >= 1)
                log("Authentication failed, show error page");
	    request.setRequestURI(contextPath + config.getErrorPage());
	    return (true);	// Display the error page in the usual manner
	}


	// Restore this request and redirect to the original request URI
        session = getSession(request, true);
        if (debug >= 1)
            log("restore request from session '" + session.getId() + "'");
        register(request, response, principal, Constants.FORM_METHOD);
	if (restoreRequest(request, session)) {
            if (debug >= 1)
                log("Proceed to restored request");
	    return (true);		// Perform the original request
	} else {
            if (debug >= 1)
                log("Restore of original request failed");
	    hres.sendError(HttpServletResponse.SC_BAD_REQUEST);
            //	    hres.flushBuffer();
	    return (false);
	}

    }


    // -------------------------------------------------------- Private Methods


    /**
     * Restore the original request from information stored in our session.
     * If the original request is no longer present (because the session
     * timed out), return <code>false</code>; otherwise, return
     * <code>true</code>.
     *
     * @param request The request to be restored
     * @param session The session containing the saved information
     */
    private boolean restoreRequest(HttpRequest request, Session session) {

	// Retrieve and remove the SavedRequest object from our session
	SavedRequest saved = (SavedRequest)
	    session.getSession().getAttribute(Constants.FORM_KEY);
	session.getSession().removeAttribute(Constants.FORM_KEY);
	if (saved == null)
	    return (false);

	// Modify our current request to reflect the original one
	request.clearCookies();
	Iterator cookies = saved.getCookies();
	while (cookies.hasNext()) {
	    request.addCookie((Cookie) cookies.next());
	}
	request.clearHeaders();
	Iterator names = saved.getHeaderNames();
	while (names.hasNext()) {
	    String name = (String) names.next();
	    Iterator values = saved.getHeaderValues(name);
	    while (values.hasNext()) {
	        request.addHeader(name, (String) values.next());
	    }
	}
	request.clearLocales();
	Iterator locales = saved.getLocales();
	while (locales.hasNext()) {
	    request.addLocale((Locale) locales.next());
	}
	request.clearParameters();
	Iterator paramNames = saved.getParameterNames();
	while (paramNames.hasNext()) {
	    String paramName = (String) paramNames.next();
	    String paramValues[] =
	      (String[]) saved.getParameterValues(paramName);
	    request.addParameter(paramName, paramValues);
	}
	request.setMethod(saved.getMethod());
	request.setQueryString(saved.getQueryString());
	request.setRequestURI(saved.getRequestURI());
	return (true);

    }


    /**
     * Save the original request information into our session.
     *
     * @param request The request to be saved
     * @param session The session to contain the saved information
     */
    private void saveRequest(HttpRequest request, Session session) {

	// Create and populate a SavedRequest object for this request
	HttpServletRequest hreq = (HttpServletRequest) request.getRequest();
	SavedRequest saved = new SavedRequest();
	Cookie cookies[] = hreq.getCookies();
	if (cookies != null) {
	    for (int i = 0; i < cookies.length; i++)
		saved.addCookie(cookies[i]);
	}
	Enumeration names = hreq.getHeaderNames();
	while (names.hasMoreElements()) {
	    String name = (String) names.nextElement();
	    Enumeration values = hreq.getHeaders(name);
	    while (values.hasMoreElements()) {
		String value = (String) values.nextElement();
		saved.addHeader(name, value);
	    }
	}
	Enumeration locales = hreq.getLocales();
	while (locales.hasMoreElements()) {
	    Locale locale = (Locale) locales.nextElement();
	    saved.addLocale(locale);
	}
	Map parameters = hreq.getParameterMap();
	Iterator paramNames = parameters.keySet().iterator();
	while (paramNames.hasNext()) {
	    String paramName = (String) paramNames.next();
	    String paramValues[] = (String[]) parameters.get(paramName);
	    saved.addParameter(paramName, paramValues);
	}
	saved.setMethod(hreq.getMethod());
	saved.setQueryString(hreq.getQueryString());
	saved.setRequestURI(hreq.getRequestURI());

	// Stash the SavedRequest in our session for later use
	session.getSession().setAttribute(Constants.FORM_KEY, saved);

    }


}
