/*
 * CompressionFilter.java
 * $Header: /home/cvs/jakarta-tomcat-4.0/webapps/examples/WEB-INF/classes/compressionFilters/CompressionFilter.java,v 1.3 2001/03/22 17:49:24 amyroh Exp $
 * $Revision: 1.3 $
 * $Date: 2001/03/22 17:49:24 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package compressionFilters;

import java.io.IOException;
import java.util.Enumeration;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Implementation of <code>javax.servlet.Filter</code> used to compress
 * the ServletResponse if it is bigger than a threshold.
 *
 * @author Amy Roh
 * @version $Revision: 1.3 $, $Date: 2001/03/22 17:49:24 $
 */

public class CompressionFilter implements Filter{

  /**
   * The filter configuration object we are associated with.  If this value
   * is null, this filter instance is not currently configured.
   */
  private FilterConfig config = null;

  /**
   * The threshold number to compress
   */
  protected int compressionThreshold;

  /**
   * Debug level for this filter
   */
  private int debug = 0;

  /**
   * Place this filter into service.
   *
   * @param filterConfig The filter configuration object
   */

  public void init(FilterConfig filterConfig) {

    config = filterConfig;
    if (filterConfig != null) {
      String str = filterConfig.getInitParameter("compressionThreshold");
      if (str!=null) {
        compressionThreshold = Integer.parseInt(str);
      }
      else compressionThreshold = 0;
      String value = filterConfig.getInitParameter("debug");
      if (value!=null) {
	      debug = Integer.parseInt(value);
      }
      else debug = 0;
    } else {
      compressionThreshold = 0;
    }

  }

  /**
   * Take this filter out of service.
   */

  public void destroy() {

    this.config = null;

  }


 /**
	* The <code>doFilter</code> method of the Filter is called by the container
	* each time a request/response pair is passed through the chain due
	* to a client request for a resource at the end of the chain. The FilterChain passed into this
	* method allows the Filter to pass on the request and response to the next entity in the
	* chain.<p>
	* This method first examines the request to check whether the client support compression. <br>
	* It simply just pass the request and response if there is no support for compression.<br>
	* If the compression support is available, it creates a CompressionServletResponseWrapper object which
	* compresses the content and modifies the header if the content length is big enough.
	* It then invokes the next entity in the chain using the FilterChain object (<code>chain.doFilter()</code>), <br>
	**/

  public void doFilter ( ServletRequest request, ServletResponse response, FilterChain chain ) throws IOException, ServletException {

    if (debug > 0) {
      System.out.println("@doFilter");
    }
    boolean supportCompression = false;
    if (request instanceof HttpServletRequest) {
      Enumeration e = ((HttpServletRequest)request).getHeaders("Accept-Encoding");
      while (e.hasMoreElements()) {
        String name = (String)e.nextElement();
        if (name.indexOf("gzip") != -1) {
          if (debug > 0) {
            System.out.println("supports compression");
          }
          supportCompression = true;
        } else {
          if (debug > 0) {
            System.out.println("no suuport for compresion");
          }
        }
      }
    }
    if (!supportCompression) {
      if (debug > 0) {
        System.out.println("doFilter gets called wo compression");
      }
      chain.doFilter(request, response);
      return;
    } else {
      if (response instanceof HttpServletResponse) {
        CompressionServletResponseWrapper wrappedResponse = new CompressionServletResponseWrapper((HttpServletResponse)response);
        wrappedResponse.setCompressionThreshold(compressionThreshold);
        if (debug > 0) {
          System.out.println("doFilter gets called with compression");
        }
        chain.doFilter(request, wrappedResponse);
        return;
      }
    }
  }
}

