/*
 * $Header: /home/cvspublic/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/connector/http/HttpResponseStream.java,v 1.3 2000/06/04 23:18:47 remm Exp $
 * $Revision: 1.3 $
 * $Date: 2000/06/04 23:18:47 $
 *
 * ====================================================================
 * 
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 


package org.apache.tomcat.connector.http;

import java.io.IOException;
import org.apache.tomcat.Response;
import org.apache.tomcat.connector.ResponseStream;

/**
 * Response stream for the HTTP/1.1 connector. This stream will automatically
 * chunk the answer if using HTTP/1.1 and no Content-Length has been properly
 * set.
 *
 * @author <a href="mailto:remm@apache.org">Remy Maucherat</a>
 */
public final class HttpResponseStream extends ResponseStream {


    // ----------------------------------------------------------- Constructors


    private static final int MAX_CHUNK_SIZE = 4096;


    private static final String CRLF = "\r\n";


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a servlet output stream associated with the specified Request.
     *
     * @param response The associated response
     */
    public HttpResponseStream(HttpResponseImpl response) {

	super(response);
        this.useChunking = (response.isChunkingAllowed()
                            && response.getContentLength() == -1
                            && response.getStatus() != 206
                            && response.getStatus() != 304);
        if (this.useChunking)
            response.addHeader("Transfer-Encoding", "chunked");

    }


    // ----------------------------------------------------- Instance Variables


    /**
     * True if chunking is allowed.
     */
    private boolean useChunking;


    /**
     * True if printing a chunk.
     */
    private boolean writingChunk;


    // -------------------------------------------- ServletOutputStream Methods


    /**
     * Write the specified byte to our output stream.
     *
     * @param b The byte to be written
     *
     * @exception IOException if an input/output error occurs
     */
    public void write(int b)
        throws IOException {
        if (useChunking && !writingChunk) {
            writingChunk = true;
            print("1\r\n");
            super.write(b);
            println();
            writingChunk = false;
        } else {
            super.write(b);
        }
    }


    /**
     * Write the specified byte array.
     */
    public void write(byte[] b, int off, int len)
        throws IOException {
        if (useChunking && !writingChunk) {
            writingChunk = true;
            print(Integer.toHexString(len) + CRLF);
            super.write(b, off, len);
            println();
            writingChunk = false;
        } else {
            super.write(b, off, len);
        }
    }


    /**
     * Close this output stream, causing any buffered data to be flushed and
     * any further output data to throw an IOException.
     */
    public void close() throws IOException {

        if (useChunking) {
            // Write the final chunk.
            writingChunk = true;
            print("0\r\n\r\n");
            writingChunk = false;
        }
        super.close();

    }


}
