/*
 * $Header: /home/cvspublic/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/core/ContainerBase.java,v 1.17 2000/05/22 20:11:09 craigmcc Exp $
 * $Revision: 1.17 $
 * $Date: 2000/05/22 20:11:09 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */


package org.apache.tomcat.core;


import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import javax.servlet.ServletException;
import org.apache.tomcat.Container;
import org.apache.tomcat.ContainerEvent;
import org.apache.tomcat.ContainerListener;
import org.apache.tomcat.Lifecycle;
import org.apache.tomcat.LifecycleEvent;
import org.apache.tomcat.LifecycleException;
import org.apache.tomcat.LifecycleListener;
import org.apache.tomcat.Loader;
import org.apache.tomcat.Logger;
import org.apache.tomcat.Manager;
import org.apache.tomcat.Mapper;
import org.apache.tomcat.Pipeline;
import org.apache.tomcat.Realm;
import org.apache.tomcat.Request;
import org.apache.tomcat.Resources;
import org.apache.tomcat.Response;
import org.apache.tomcat.Valve;
import org.apache.tomcat.util.LifecycleSupport;
import org.apache.tomcat.util.StringManager;


/**
 * Abstract implementation of the <b>Container</b> interface, providing common
 * functionality required by nearly every implementation.  Classes extending
 * this base class must implement <code>getInfo()</code>, and may implement
 * a replacement for <code>invoke()</code>.
 * <p>
 * If a subclass prefers to use the default version of <code>invoke()</code>
 * with pipeline support, it should instantiate a Valve instance containing
 * the default behavior desired, and assign it with <code>setBasic()</code>
 * in a constructor.  In that way, any and all configured Valves added to the
 * pipeline will preceed the basic Valve.
 * <p>
 * This implementation fires property change events, per the JavaBeans design
 * pattern, for changes in singleton properties.  In addition, it fires the
 * following <code>ContainerEvent</code> events to listeners who register
 * themselves with <code>addContainerListener()</code>:
 * <table border=1>
 *   <tr>
 *     <th>Type</th>
 *     <th>Data</th>
 *     <th>Description</th>
 *   </tr>
 *   <tr>
 *     <td align=center><code>addChild</code></td>
 *     <td align=center><code>Container</code></td>
 *     <td>Child container added to this Container.</td>
 *   </tr>
 *   <tr>
 *     <td align=center><code>addValve</code></td>
 *     <td align=center><code>Valve</code></td>
 *     <td>Valve added to this Container.</td>
 *   </tr>
 *   <tr>
 *     <td align=center><code>removeChild</code></td>
 *     <td align=center><code>Container</code></td>
 *     <td>Child container removed from this Container.</td>
 *   </tr>
 *   <tr>
 *     <td align=center><code>removeValve</code></td>
 *     <td align=center><code>Valve</code></td>
 *     <td>Valve removed from this Container.</td>
 *   </tr>
 *   <tr>
 *     <td align=center><code>start</code></td>
 *     <td align=center><code>null</code></td>
 *     <td>Container was started.</td>
 *   </tr>
 *   <tr>
 *     <td align=center><code>stop</code></td>
 *     <td align=center><code>null</code></td>
 *     <td>Container was stopped.</td>
 *   </tr>
 * </table>
 * Subclasses that fire additional events should document them in the
 * class comments of the implementation class.
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.17 $ $Date: 2000/05/22 20:11:09 $
 */

public abstract class ContainerBase
    implements Container, Lifecycle, Pipeline {


    // ----------------------------------------------------- Instance Variables


    /**
     * The Valve that implements the basic behavior of this Container, if any.
     */
    protected Valve basic = null;


    /**
     * The child Containers belonging to this Container, keyed by name.
     */
    protected Hashtable children = new Hashtable();


    /**
     * The debugging detail level for this component.
     */
    protected int debug = 0;


    /**
     * The first Valve in the pipeline associated with this Container.
     */
    protected Valve first = null;


    /**
     * The lifecycle event support for this component.
     */
    protected LifecycleSupport lifecycle = new LifecycleSupport(this);


    /**
     * The container event listeners for this Container.
     */
    protected Vector listeners = new Vector();


    /**
     * The Loader implementation with which this Container is associated.
     */
    protected Loader loader = null;


    /**
     * The Logger implementation with which this Container is associated.
     */
    protected Logger logger = null;


    /**
     * The Manager implementation with which this Container is associated.
     */
    protected Manager manager = null;


    /**
     * The one and only Mapper associated with this Container, if any.
     */
    protected Mapper mapper = null;


    /**
     * The set of Mappers associated with this Container, keyed by protocol.
     */
    protected Hashtable mappers = new Hashtable();


    /**
     * The Java class name of the default Mapper class for this Container.
     */
    protected String mapperClass = null;


    /**
     * The human-readable name of this Container.
     */
    protected String name = null;


    /**
     * The parent Container to which this Container is a child.
     */
    protected Container parent = null;


    /**
     * The Realm with which this Container is associated.
     */
    protected Realm realm = null;


    /**
     * The Resources object with which this Container is associated.
     */
    protected Resources resources = null;


    /**
     * The string manager for this package.
     */
    protected static StringManager sm =
	StringManager.getManager(Constants.Package);


    /**
     * Has this component been started?
     */
    protected boolean started = false;


    /**
     * The property change support for this component.
     */
    protected PropertyChangeSupport support = new PropertyChangeSupport(this);


    // ------------------------------------------------------------- Properties


    /**
     * Return the Valve that provides the basic functionality for this
     * Container, if any.
     */
    public Valve getBasic() {

	return (basic);

    }


    /**
     * Set the Valve that provides the basic functionality for this
     * Container, if any.
     *
     * @param valve The new basic Valve
     */
    public synchronized void setBasic(Valve valve) {

	// Change components if necessary
	Valve oldBasic = this.basic;
	if (oldBasic == valve)
	    return;
	this.basic = valve;

	// Stop the old component if necessary and remove it from the pipeline
	if (oldBasic != null) {
	    Valve previous = getLast();
	    if (previous != null)
		previous.setNext(null);
	    oldBasic.setContainer(null);
	}
	if (started && (oldBasic != null) && (oldBasic instanceof Lifecycle)) {
	    try {
		((Lifecycle) oldBasic).stop();
	    } catch (LifecycleException e) {
		log("ContainerBase.setBasic: stop: ", e);
	    }
	}

	// Start the new component if necessary and link it into the pipeline
	if (this.basic != null) {
	    this.basic.setContainer((Container) this);
	    this.basic.setNext(null);
	}
	if (started && (valve != null) && (valve instanceof Lifecycle)) {
	    try {
		((Lifecycle) valve).start();
	    } catch (LifecycleException e) {
		log("ContainerBase.setBasic: start: ", e);
	    }
	}
	if (this.basic != null) {
	    Valve previous = getLast();
	    if (previous != null)
		previous.setNext(this.basic);
	}

	// Report this property change to interested listeners
	support.firePropertyChange("basic", oldBasic, this.basic);

    }


    /**
     * Return the debugging detail level for this component.
     */
    public int getDebug() {

	return (this.debug);

    }


    /**
     * Set the debugging detail level for this component.
     *
     * @param debug The new debugging detail level
     */
    public void setDebug(int debug) {

	int oldDebug = this.debug;
	this.debug = debug;
	support.firePropertyChange("debug", new Integer(oldDebug),
				   new Integer(this.debug));

    }


    /**
     * Return descriptive information about this Container implementation and
     * the corresponding version number, in the format
     * <code>&lt;description&gt;/&lt;version&gt;</code>.
     */
    public abstract String getInfo();


    /**
     * Return the last configured Valve (other than the basic Valve, if any)
     * configured in the pipeline for this Container, if any; otherwise
     * return <code>null</code>.
     */
    public Valve getLast() {

	if (first == null)
	    return (null);
	Valve next = first;
	while ((next.getNext() != null) && (next.getNext() != basic))
	    next = next.getNext();
	return (next);

    }


    /**
     * Return the Loader with which this Container is associated.  If there is
     * no associated Loader, return the Loader associated with our parent
     * Container (if any); otherwise, return <code>null</code>.
     */
    public Loader getLoader() {

	if (loader != null)
	    return (loader);
	if (parent != null)
	    return (parent.getLoader());
	return (null);

    }


    /**
     * Set the Loader with which this Container is associated.
     *
     * @param loader The newly associated loader
     */
    public synchronized void setLoader(Loader loader) {

	// Change components if necessary
	Loader oldLoader = this.loader;
	if (oldLoader == loader)
	    return;
	this.loader = loader;

	// Stop the old component if necessary
	if (started && (oldLoader != null) &&
	    (oldLoader instanceof Lifecycle)) {
	    try {
		((Lifecycle) oldLoader).stop();
	    } catch (LifecycleException e) {
		log("ContainerBase.setLoader: stop: ", e);
	    }
	}

	// Start the new component if necessary
	if (loader != null)
	    loader.setContainer(this);
	if (started && (loader != null) &&
	    (loader instanceof Lifecycle)) {
	    try {
		((Lifecycle) loader).start();
	    } catch (LifecycleException e) {
		log("ContainerBase.setLoader: start: ", e);
	    }
	}

	// Report this property change to interested listeners
	support.firePropertyChange("loader", oldLoader, this.loader);

    }


    /**
     * Return the Logger with which this Container is associated.  If there is
     * no associated Logger, return the Logger associated with our parent
     * Container (if any); otherwise return <code>null</code>.
     */
    public Logger getLogger() {

	if (logger != null)
	    return (logger);
	if (parent != null)
	    return (parent.getLogger());
	return (null);

    }


    /**
     * Set the Logger with which this Container is associated.
     *
     * @param logger The newly associated Logger
     */
    public synchronized void setLogger(Logger logger) {

	// Change components if necessary
	Logger oldLogger = this.logger;
	if (oldLogger == logger)
	    return;
	this.logger = logger;

	// Stop the old component if necessary
	if (started && (oldLogger != null) &&
	    (oldLogger instanceof Lifecycle)) {
	    try {
		((Lifecycle) oldLogger).stop();
	    } catch (LifecycleException e) {
		log("ContainerBase.setLogger: stop: ", e);
	    }
	}

	// Start the new component if necessary
	if (logger != null)
	    logger.setContainer(this);
	if (started && (logger != null) &&
	    (logger instanceof Lifecycle)) {
	    try {
		((Lifecycle) logger).start();
	    } catch (LifecycleException e) {
		log("ContainerBase.setLogger: start: ", e);
	    }
	}

	// Report this property change to interested listeners
	support.firePropertyChange("logger", oldLogger, this.logger);

    }


    /**
     * Return the Manager with which this Container is associated.  If there is
     * no associated Manager, return the Manager associated with our parent
     * Container (if any); otherwise return <code>null</code>.
     */
    public Manager getManager() {

	if (manager != null)
	    return (manager);
	if (parent != null)
	    return (parent.getManager());
	return (null);

    }


    /**
     * Set the Manager with which this Container is associated.
     *
     * @param manager The newly associated Manager
     */
    public synchronized void setManager(Manager manager) {

	// Change components if necessary
	Manager oldManager = this.manager;
	if (oldManager == manager)
	    return;
	this.manager = manager;

	// Stop the old component if necessary
	if (started && (oldManager != null) &&
	    (oldManager instanceof Lifecycle)) {
	    try {
		((Lifecycle) oldManager).stop();
	    } catch (LifecycleException e) {
		log("ContainerBase.setManager: stop: ", e);
	    }
	}

	// Start the new component if necessary
	if (manager != null)
	    manager.setContainer(this);
	if (started && (manager != null) &&
	    (manager instanceof Lifecycle)) {
	    try {
		((Lifecycle) manager).start();
	    } catch (LifecycleException e) {
		log("ContainerBase.setManager: start: ", e);
	    }
	}

	// Report this property change to interested listeners
	support.firePropertyChange("manager", oldManager, this.manager);

    }


    /**
     * Return a name string (suitable for use by humans) that describes this
     * Container.  Within the set of child containers belonging to a particular
     * parent, Container names must be unique.
     */
    public String getName() {

	return (name);

    }


    /**
     * Set a name string (suitable for use by humans) that describes this
     * Container.  Within the set of child containers belonging to a particular
     * parent, Container names must be unique.
     *
     * @param name New name of this container
     *
     * @exception IllegalStateException if this Container has already been
     *  added to the children of a parent Container (after which the name
     *  may not be changed)
     */
    public void setName(String name) {

	String oldName = this.name;
	this.name = name;
	support.firePropertyChange("name", oldName, this.name);

    }


    /**
     * Return the Container for which this Container is a child, if there is
     * one.  If there is no defined parent, return <code>null</code>.
     */
    public Container getParent() {

	return (parent);

    }


    /**
     * Set the parent Container to which this Container is being added as a
     * child.  This Container may refuse to become attached to the specified
     * Container by throwing an exception.
     *
     * @param container Container to which this Container is being added
     *  as a child
     *
     * @exception IllegalArgumentException if this Container refuses to become
     *  attached to the specified Container
     */
    public void setParent(Container container) {

	Container oldParent = this.parent;
	this.parent = container;
	support.firePropertyChange("parent", oldParent, this.parent);

    }


    /**
     * Return the Realm with which this Container is associated.  If there is
     * no associated Realm, return the Realm associated with our parent
     * Container (if any); otherwise return <code>null</code>.
     */
    public Realm getRealm() {

	if (realm != null)
	    return (realm);
	if (parent != null)
	    return (parent.getRealm());
	return (null);

    }


    /**
     * Set the Realm with which this Container is associated.
     *
     * @param realm The newly associated Realm
     */
    public synchronized void setRealm(Realm realm) {

	// Change components if necessary
	Realm oldRealm = this.realm;
	if (oldRealm == realm)
	    return;
	this.realm = realm;

	// Stop the old component if necessary
	if (started && (oldRealm != null) &&
	    (oldRealm instanceof Lifecycle)) {
	    try {
		((Lifecycle) oldRealm).stop();
	    } catch (LifecycleException e) {
		log("ContainerBase.setRealm: stop: ", e);
	    }
	}

	// Start the new component if necessary
	if (realm != null)
	    realm.setContainer(this);
	if (started && (realm != null) &&
	    (realm instanceof Lifecycle)) {
	    try {
		((Lifecycle) realm).start();
	    } catch (LifecycleException e) {
		log("ContainerBase.setRealm: start: ", e);
	    }
	}

	// Report this property change to interested listeners
	support.firePropertyChange("realm", oldRealm, this.realm);

    }


    /**
     * Return the Resources with which this Container is associated.  If there
     * is no associated Resources object, return the Resources associated with
     * our parent Container (if any); otherwise return <code>null</code>.
     */
    public Resources getResources() {

	if (resources != null)
	    return (resources);
	if (parent != null)
	    return (parent.getResources());
	return (null);

    }


    /**
     * Set the Resources object with which this Container is associated.
     *
     * @param resources The newly associated Resources
     */
    public synchronized void setResources(Resources resources) {

	// Change components if necessary
	Resources oldResources = this.resources;
	if (oldResources == resources)
	    return;
	this.resources = resources;

	// Stop the old component if necessary
	if (started && (oldResources != null) &&
	    (oldResources instanceof Lifecycle)) {
	    try {
		((Lifecycle) oldResources).stop();
	    } catch (LifecycleException e) {
		log("ContainerBase.setResources: stop: ", e);
	    }
	}
	if (oldResources != null)
	    oldResources.setContainer(null);

	// Start the new component if necessary
	if (resources != null)
	    resources.setContainer(this);
	if (started && (resources != null) &&
	    (resources instanceof Lifecycle)) {
	    try {
		((Lifecycle) resources).start();
	    } catch (LifecycleException e) {
		log("ContainerBase.setResources: start: ", e);
	    }
	}

	// Report this property change to interested listeners
	support.firePropertyChange("resources", oldResources, this.resources);

    }


    // ------------------------------------------------------ Container Methods


    /**
     * Add a new child Container to those associated with this Container,
     * if supported.  Prior to adding this Container to the set of children,
     * the child's <code>setParent()</code> method must be called, with this
     * Container as an argument.  This method may thrown an
     * <code>IllegalArgumentException</code> if this Container chooses not
     * to be attached to the specified Container, in which case it is not added
     *
     * @param child New child Container to be added
     *
     * @exception IllegalArgumentException if this exception is thrown by
     *  the <code>setParent()</code> method of the child Container
     * @exception IllegalArgumentException if the new child does not have
     *  a name unique from that of existing children of this Container
     * @exception IllegalStateException if this Container does not support
     *  child Containers
     */
    public void addChild(Container child) {

	synchronized(children) {
	    if (children.get(child.getName()) != null)
		throw new IllegalArgumentException("addChild:  Child name '" +
						   child.getName() +
						   "' is not unique");
	    child.setParent((Container) this);	// May throw IAE
	    if (started && (child instanceof Lifecycle)) {
		try {
		    ((Lifecycle) child).start();
		} catch (LifecycleException e) {
		    log("ContainerBase.addChild: start: ", e);
		    throw new IllegalStateException
		        ("ContainerBase.addChild: start: " + e);
		}
	    }
	    children.put(child.getName(), child);
	    fireContainerEvent(ADD_CHILD_EVENT, child);
	}

    }


    /**
     * Add a container event listener to this component.
     *
     * @param listener The listener to add
     */
    public void addContainerListener(ContainerListener listener) {

	listeners.addElement(listener);

    }


    /**
     * Add the specified Mapper associated with this Container.
     *
     * @param mapper The corresponding Mapper implementation
     *
     * @exception IllegalArgumentException if this exception is thrown by
     *  the <code>setContainer()</code> method of the Mapper
     */
    public void addMapper(Mapper mapper) {

	synchronized(mappers) {
	    if (mappers.get(mapper.getProtocol()) != null)
		throw new IllegalArgumentException("addMapper:  Protocol '" +
						   mapper.getProtocol() +
						   "' is not unique");
	    mapper.setContainer((Container) this);	// May throw IAE
	    if (started && (mapper instanceof Lifecycle)) {
		try {
		    ((Lifecycle) mapper).start();
		} catch (LifecycleException e) {
		    log("ContainerBase.addMapper: start: ", e);
		    throw new IllegalStateException
			("ContainerBase.addMapper: start: " + e);
		}
	    }
	    mappers.put(mapper.getProtocol(), mapper);
	    if (mappers.size() == 1)
		this.mapper = mapper;
	    else
		this.mapper = null;
	    fireContainerEvent(ADD_MAPPER_EVENT, mapper);
	}

    }


    /**
     * Add a property change listener to this component.
     *
     * @param listener The listener to add
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {

	support.addPropertyChangeListener(listener);

    }


    /**
     * Return the child Container, associated with this Container, with
     * the specified name (if any); otherwise, return <code>null</code>
     *
     * @param name Name of the child Container to be retrieved
     */
    public Container findChild(String name) {

	if (name == null)
	    return (null);
	return ((Container) children.get(name));

    }


    /**
     * Return the set of children Containers associated with this Container.
     * If this Container has no children, a zero-length array is returned.
     */
    public Container[] findChildren() {

	synchronized (children) {
	    int n = children.size();
	    Container results[] = new Container[n];
	    Enumeration containers = children.elements();
	    for (int i = 0; i < n; i++)
		results[i] = (Container) containers.nextElement();
	    return (results);
	}

    }


    /**
     * Return the Mapper associated with the specified protocol, if there
     * is one.  If there is only one defined Mapper, use it for all protocols.
     * If there is no matching Mapper, return <code>null</code>.
     *
     * @param protocol Protocol for which to find a Mapper
     */
    public Mapper findMapper(String protocol) {

	if (mapper != null)
	    return (mapper);
	else
	    return ((Mapper) mappers.get(protocol));

    }


    /**
     * Return the set of Mappers associated with this Container.  If this
     * Container has no Mappers, a zero-length array is returned.
     */
    public Mapper[] findMappers() {

	synchronized (mappers) {
	    int i = 0;
	    int n = mappers.size();
	    Mapper results[] = new Mapper[n];
	    Enumeration keys = mappers.keys();
	    while (keys.hasMoreElements()) {
		String key = (String) keys.nextElement();
		results[i++] = (Mapper) mappers.get(key);
	    }
	    return (results);
	}

    }


    /**
     * Process the specified Request, to produce the corresponding Response,
     * by invoking the first Valve in our pipeline (if any), or the basic
     * Valve otherwise.
     *
     * @param request Request to be processed
     * @param response Response to be produced
     *
     * @exception IllegalStateException if neither a pipeline or a basic
     *  Valve have been configured for this Container
     * @exception IOException if an input/output error occurred while
     *  processing
     * @exception ServletException if a ServletException was thrown
     *  while processing this request
     */
    public void invoke(Request request, Response response)
	throws IOException, ServletException {

	if (first != null)
	    first.invoke(request, response);
	else if (basic != null)
	    basic.invoke(request, response);
	else
	    throw new IllegalStateException
		(sm.getString("containerBase.notConfigured"));

    }


    /**
     * Return the child Container that should be used to process this Request,
     * based upon its characteristics.  If no such child Container can be
     * identified, return <code>null</code> instead.
     *
     * @param request Request being processed
     * @param update Update the Request to reflect the mapping selection?
     */
    public Container map(Request request, boolean update) {

	// Select the Mapper we will use
	Mapper mapper = findMapper(request.getRequest().getProtocol());
	if (mapper == null)
	    return (null);

	// Use this Mapper to perform this mapping
	return (mapper.map(request, update));

    }


    /**
     * Remove an existing child Container from association with this parent
     * Container.
     *
     * @param child Existing child Container to be removed
     */
    public void removeChild(Container child) {

	synchronized(children) {
	    if (children.get(child.getName()) == null)
		return;
	    children.remove(child.getName());
	    if (started && (child instanceof Lifecycle)) {
		try {
		    ((Lifecycle) child).stop();
		} catch (LifecycleException e) {
		    log("ContainerBase.removeChild: stop: ", e);
		}
	    }
	    child.setParent(null);
	    fireContainerEvent(REMOVE_CHILD_EVENT, child);
	}

    }


    /**
     * Remove a container event listener from this component.
     *
     * @param listener The listener to remove
     */
    public void removeContainerListener(ContainerListener listener) {

	listeners.removeElement(listener);

    }


    /**
     * Remove a Mapper associated with this Container, if any.
     *
     * @param mapper The Mapper to be removed
     */
    public void removeMapper(Mapper mapper) {

	synchronized(mappers) {

	    if (mappers.get(mapper.getProtocol()) == null)
		return;
	    mappers.remove(mapper.getProtocol());
	    if (started && (mapper instanceof Lifecycle)) {
		try {
		    ((Lifecycle) mapper).stop();
		} catch (LifecycleException e) {
		    log("ContainerBase.removeMapper: stop: ", e);
		    throw new IllegalStateException
			("ContainerBase.removeMapper: stop: " + e);
		}
	    }
	    if (mappers.size() != 1)
		this.mapper = null;
	    else {
		Enumeration elements = mappers.elements();
	        this.mapper = (Mapper) elements.nextElement();
	    }
	    fireContainerEvent(REMOVE_MAPPER_EVENT, mapper);
	}

    }


    /**
     * Remove a property change listener from this component.
     *
     * @param listener The listener to remove
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {

	support.removePropertyChangeListener(listener);

    }


    // ------------------------------------------------------ Lifecycle Methods


    /**
     * Add a lifecycle event listener to this component.
     *
     * @param listener The listener to add
     */
    public void addLifecycleListener(LifecycleListener listener) {

	lifecycle.addLifecycleListener(listener);

    }


    /**
     * Remove a lifecycle event listener from this component.
     *
     * @param listener The listener to remove
     */
    public void removeLifecycleListener(LifecycleListener listener) {

	lifecycle.removeLifecycleListener(listener);

    }


    /**
     * Prepare for active use of the public methods of this Component.
     *
     * @exception IllegalStateException if this component has already been
     *  started
     * @exception LifecycleException if this component detects a fatal error
     *  that prevents it from being started
     */
    public synchronized void start() throws LifecycleException {

	// Validate and update our current component state
	if (started)
	    throw new LifecycleException
		(sm.getString("containerBase.alreadyStarted"));
	lifecycle.fireLifecycleEvent(START_EVENT, null);
	started = true;

	// Start our subordinate components, if any
	if ((loader != null) && (loader instanceof Lifecycle))
	    ((Lifecycle) loader).start();
	if ((logger != null) && (logger instanceof Lifecycle))
	    ((Lifecycle) logger).start();
	if ((manager != null) && (manager instanceof Lifecycle))
	    ((Lifecycle) manager).start();
	if ((realm != null) && (realm instanceof Lifecycle))
	    ((Lifecycle) realm).start();
	if ((resources != null) && (resources instanceof Lifecycle))
	    ((Lifecycle) resources).start();

	// Add a default Mapper if none have been defined
	addDefaultMapper(this.mapperClass);

	// Start our Mappers, if any
	Mapper mappers[] = findMappers();
	for (int i = 0; i < mappers.length; i++) {
	    if (mappers[i] instanceof Lifecycle)
		((Lifecycle) mappers[i]).start();
	}

	// Start our child containers, if any
	Container children[] = findChildren();
	for (int i = 0; i < children.length; i++) {
	    if (children[i] instanceof Lifecycle)
		((Lifecycle) children[i]).start();
	}

	// Start the Valves in our pipeline (including the basic), if any
	Valve current = first;
	while (current != null) {
	    if (current instanceof Lifecycle)
		((Lifecycle) current).start();
	    current = current.getNext();
	}

    }


    /**
     * Gracefully shut down active use of the public methods of this Component.
     *
     * @exception IllegalStateException if this component has not been started
     * @exception LifecycleException if this component detects a fatal error
     *  that needs to be reported
     */
    public synchronized void stop() throws LifecycleException {

	// Validate and update our current component state
	if (!started)
	    throw new LifecycleException
		(sm.getString("containerBase.notStarted"));
	lifecycle.fireLifecycleEvent(STOP_EVENT, null);
	started = false;

	// Stop the Valves in our pipeline (including the basic), if any
	Valve current = basic;
	if (current == null)
	    current = getLast();
	while (current != null) {
	    if (current instanceof Lifecycle)
		((Lifecycle) current).stop();
	    current = current.getPrevious();
	}

	// Stop our child containers, if any
	Container children[] = findChildren();
	for (int i = 0; i < children.length; i++) {
	    if (children[(children.length-1)-i] instanceof Lifecycle)
		((Lifecycle) children[(children.length-1)-i]).stop();
	}

	// Stop our Mappers, if any
	Mapper mappers[] = findMappers();
	for (int i = 0; i < mappers.length; i++) {
	    if (mappers[(mappers.length-1)-i] instanceof Lifecycle)
		((Lifecycle) mappers[(mappers.length-1)-i]).stop();
	}

	// Stop our subordinate components, if any
	if ((resources != null) && (resources instanceof Lifecycle))
	    ((Lifecycle) resources).stop();
	if ((realm != null) && (realm instanceof Lifecycle))
	    ((Lifecycle) realm).stop();
	if ((manager != null) && (manager instanceof Lifecycle))
	    ((Lifecycle) manager).stop();
	if ((logger != null) && (logger instanceof Lifecycle))
	    ((Lifecycle) logger).stop();
	if ((loader != null) && (loader instanceof Lifecycle))
	    ((Lifecycle) loader).stop();

    }


    // ------------------------------------------------------- Pipeline Methods


    /**
     * Add a new Valve to the end of the pipeline associated with this
     * Container.  Prior to adding the Valve, the Valve's
     * <code>setContainer</code> method must be called, with this Container
     * as an argument.  The method may throw an
     * <code>IllegalArgumentException</code> if this Valve chooses not to
     * be associated with this Container, or <code>IllegalStateException</code>
     * if it is already associated with a different Container.
     *
     * @param valve Valve to be added
     *
     * @exception IllegalArgumentException if this Container refused to
     *  accept the specified Valve
     * @exception IllegalArgumentException if the specifie Valve refuses to be
     *  associated with this Container
     * @exception IllegalStateException if the specified Valve is already
     *  associated with a different Container
     */
    public synchronized void addValve(Valve valve) {

	// Start the new component if necessary
	valve.setContainer((Container) this);
	valve.setNext(basic);
	if (started && (valve != null) && (valve instanceof Lifecycle)) {
	    try {
		((Lifecycle) valve).start();
	    } catch (LifecycleException e) {
		log("ContainerBase.addValve: start: ", e);
	    }
	}

	// Link the new component into the pipeline
	if (basic == null) {
	    Valve last = getLast();
	    if (last == null) {
		valve.setPrevious(null);
		first = valve;
	    } else {
		valve.setPrevious(last);
		last.setNext(valve);
	    }
	} else {
	    Valve previous = basic.getPrevious();
	    if (previous == null) {
		valve.setPrevious(null);
		first = valve;
	    } else {
		valve.setPrevious(previous);
		previous.setNext(valve);
	    }
	}

	// Report this pipeline change to interested listeners
	fireContainerEvent(ADD_VALVE_EVENT, valve);

    }


    /**
     * Return the first Valve in the pipeline associated with this Container.
     * If there are no such Valves, <code>null</code> is returned.
     */
    public Valve findValves() {

	return (first);

    }


    /**
     * Remove the specified Valve from the pipeline associated with this
     * Container, if it is found; otherwise, do nothing.
     *
     * @param valve Valve to be removed
     */
    public synchronized void removeValve(Valve valve) {

	Valve current = first;
	while (current != null) {
	    if (current != valve) {
		current = current.getNext();
		continue;
	    }
	    Valve previous = current.getPrevious();
	    Valve next = current.getNext();
	    if (previous == null) {
		if (next == null) {
		    first = null;
		} else {
		    next.setPrevious(null);
		    first = next;
		}
	    } else {
		previous.setNext(next);
		if (next != null)
		    next.setPrevious(previous);
	    }
	    if (started && (valve != null) && (valve instanceof Lifecycle)) {
		try {
		    ((Lifecycle) valve).stop();
		} catch (LifecycleException e) {
		    log("ContainerBase.removeValve: stop: ", e);
		}
	    }
	    fireContainerEvent(REMOVE_VALVE_EVENT, valve);
	    break;
	}

    }


    // ------------------------------------------------------ Protected Methods


    /**
     * Add a default Mapper implementation if none have been configured
     * explicitly.
     *
     * @param mapperClass Java class name of the default Mapper
     */
    protected void addDefaultMapper(String mapperClass) {

	// Do we need a default Mapper?
	if (mapperClass == null)
	    return;
	if (mappers.size() >= 1)
	    return;

	// Instantiate and add a default Mapper
	try {
	    Class clazz = Class.forName(mapperClass);
	    Mapper mapper = (Mapper) clazz.newInstance();
	    mapper.setProtocol("http");
	    addMapper(mapper);
	} catch (Exception e) {
	    log(sm.getString("containerBase.addDefaultMapper", mapperClass),
		e);
	}

    }


    /**
     * Notify all container event listeners that a particular event has
     * occurred for this Container.  The default implementation performs
     * this notification synchronously using the calling thread.
     *
     * @param type Event type
     * @param data Event data
     */
    protected void fireContainerEvent(String type, Object data) {

	ContainerEvent event = new ContainerEvent(this, type, data);
	Vector list = null;
	synchronized (this) {
	    list = (Vector) listeners.clone();
	}
	for (int i = 0; i < list.size(); i++)
	    ((ContainerListener) list.elementAt(i)).containerEvent(event);

    }


    /**
     * Log the specified message to our current Logger (if any).
     *
     * @param message Message to be logged
     */
    protected void log(String message) {

	Logger logger = getLogger();
	if (logger != null)
	    logger.log(logName() + ": " + message);
	else
	    System.out.println(logName() + ": " + message);

    }


    /**
     * Log the specified message and exception to our current Logger
     * (if any).
     *
     * @param message Message to be logged
     * @param throwable Related exception
     */
    protected void log(String message, Throwable throwable) {

	Logger logger = getLogger();
	if (logger != null)
	    logger.log(logName() + ": " + message, throwable);
	else {
	    System.out.println(logName() + ": " + message + ": " + throwable);
	    throwable.printStackTrace(System.out);
	}

    }


    /**
     * Return the abbreviated name of this container for logging messsages
     */
    protected String logName() {

	String className = this.getClass().getName();
	int period = className.lastIndexOf(".");
	if (period >= 0)
	    className = className.substring(period + 1);
	return (className + "[" + getName() + "]");

    }


}
