/*
 * $Header: /home/cvspublic/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/deploy/SecurityConstraint.java,v 1.2 2000/05/03 22:58:27 craigmcc Exp $
 * $Revision: 1.2 $
 * $Date: 2000/05/03 22:58:27 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 


package org.apache.tomcat.deploy;


import java.util.Enumeration;
import java.util.Hashtable;


/**
 * Representation of a security constraint element for a web application,
 * as represented in a <code>&lt;security-constraint&gt;</code> element in the
 * deployment descriptor.
 * <p>
 * <b>WARNING</b>:  The property setting methods in this class are for use by
 * the application logic that parses a web application deployment descriptor.
 * They should not be called by a Context implementation (or an associated
 * Valve or Interceptor that implements the authentication and authorization
 * constraints expressed here).
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.2 $ $Date: 2000/05/03 22:58:27 $
 */

public final class SecurityConstraint {


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a new security constraint instance with default values.
     */
    public SecurityConstraint() {

	super();

    }


    // ----------------------------------------------------- Instance Variables


    /**
     * The set of roles permitted to access resources protected by this
     * security constraint.
     */
    private String authRoles[] = new String[0];


    /**
     * The set of web resource collections protected by this security
     * constraint, keyed by resource collection name.
     */
    private Hashtable collections = new Hashtable();


    /**
     * The URL patterns that are parts of the web resource collections
     * managed by this security constraint.  The key is the pattern itself,
     * while the value is the corresponding collection.
     */
    private Hashtable patterns = new Hashtable();


    /**
     * The user data constraint for this security constraint.  Must be NONE,
     * INTEGRAL, or CONFIDENTIAL.
     */
    private String userConstraint = "NONE";


    // ------------------------------------------------------------- Properties


    /**
     * Return the user data constraint for this security constraint.
     */
    public String getUserConstraint() {

	return (userConstraint);

    }


    /**
     * Set the user data constraint for this security constraint.
     *
     * @param userConstraint The new user data constraint
     */
    public void setUserConstraint(String userConstraint) {

	if (userConstraint != null)
	    this.userConstraint = userConstraint;

    }


    // --------------------------------------------------------- Public Methods


    /**
     * Add an authorization role, which is a role name that will be
     * permitted access to the resources protected by this security constraint.
     *
     * @param role Role name to be added
     */
    public void addAuthRole(String role) {

	if (role == null)
	    return;
	synchronized (authRoles) {
	    if (findAuthRole(role))
		return;
	    String newRoles[] = new String[authRoles.length + 1];
	    for (int i = 0; i < authRoles.length; i++)
		newRoles[i] = authRoles[i];
	    newRoles[newRoles.length - 1] = role;
	    authRoles = newRoles;
	}

    }


    /**
     * Add a new web resource collection to those protected by this
     * security constraint.
     *
     * @param collection The new web resource collection
     */
    public void addCollection(SecurityCollection collection) {

	collections.put(collection.getName(), collection);

    }


    /**
     * Create and return a web resource collection with default values.
     *
     * @param name Name of the new web resource collection
     */
    public SecurityCollection createCollection(String name) {

	return (new SecurityCollection(this, name));

    }


    /**
     * Return <code>true</code> if the specified role is permitted access to
     * the resources protected by this security constraint.
     *
     * @param role Role name to be checked
     */
    public boolean findAuthRole(String role) {

	if (role == null)
	    return (false);
	synchronized (authRoles) {
	    for (int i = 0; i < authRoles.length; i++) {
		if (role.equals(authRoles[i]))
		    return (true);
	    }
	}
	return (false);

    }


    /**
     * Return the set of roles that are permitted access to the resources
     * protected by this security constraint.  If none have been defined,
     * a zero-length array is returned (which implies that all authenticated
     * users are permitted access).
     */
    public String[] findAuthRoles() {

	return (authRoles);	// Assumption - caller will not modify

    }


    /**
     * Return the web resource collection for the specified name, if any;
     * otherwise, return <code>null</code>.
     *
     * @param name Web resource collection name to return
     */
    public SecurityCollection findCollection(String name) {

	return ((SecurityCollection) collections.get(name));

    }


    /**
     * Return the names of all web resource collections protected by this
     * security constraint.  If there are none, a zero-length array is
     * returned.
     */
    public String[] findCollections() {

	synchronized (collections) {
	    String results[] = new String[collections.size()];
	    Enumeration names = collections.keys();
	    for (int i = 0; i < results.length; i++)
		results[i] = (String) names.nextElement();
	    return (results);
	}

    }


    /**
     * Return <code>true</code> if the specified context-relative URI (and
     * associated HTTP method) are protected by this security constraint.
     *
     * @param uri Context-relative URI to check
     * @param method Request method being used
     */
    public boolean included(String uri, String method) {

	// Check all defined patterns
	SecurityCollection collection = null;
	Enumeration patterns = this.patterns.keys();
	while (patterns.hasMoreElements()) {
	    String pattern = (String) patterns.nextElement();
	    if (matchPattern(uri, pattern)) {
		collection = (SecurityCollection) this.patterns.get(pattern);
		break;
	    }
	}

	// Match on HTTP request method as well
	if (collection == null)
	    return (false);
	else
	    return (collection.findMethod(method));

    }


    /**
     * Remove the specified role from the set of roles permitted to access
     * the resources protected by this security constraint.
     *
     * @param role Role name to be removed
     */
    public void removeAuthRole(String role) {

	if (role == null)
	    return;
	synchronized (authRoles) {
	    if (!findAuthRole(role))
		return;
	    String newRoles[] = new String[authRoles.length - 1];
	    int j = 0;
	    for (int i = 0; i < authRoles.length; i++) {
		if (role.equals(authRoles[i]))
		    continue;
		newRoles[j++] = authRoles[i];
	    }
	    authRoles = newRoles;
	}

    }


    /**
     * Remove the specified web resource collection from those protected by
     * this security constraint.
     *
     * @param collection Web resource collection to be removed
     */
    public void removeCollection(SecurityCollection collection) {

	collections.remove(collection.getName());

    }


    /**
     * Return a String representation of this security constraint.
     */
    public String toString() {

        StringBuffer sb = new StringBuffer("SecurityConstraint[");
	int n = 0;
	Enumeration names = collections.keys();
	while (names.hasMoreElements()) {
	    String name = (String) names.nextElement();
	    if (n > 0)
	        sb.append(",");
	    sb.append("(");
	    sb.append(name);
	    sb.append(")");
	    n++;
	}
	sb.append("]");
	return (sb.toString());

    }


    // ------------------------------------------------ Package Private Methods


    /**
     * Add a URL pattern that is part of one of our constituent web resource
     * collections.
     *
     * @param pattern URL pattern to be added
     * @param collection The corresponding collection
     */
    void addPattern(String pattern, SecurityCollection collection) {

	patterns.put(pattern, collection);

    }


    /**
     * Remove a URL pattern that was part of one of our constituent web
     * resource collections.
     *
     * @param pattern URL pattern to be removed
     */
    void removePattern(String pattern) {

	patterns.remove(pattern);

    }


    // -------------------------------------------------------- Private Methods


    /**
     * Does the specified request path match the specified URL pattern?
     *
     * XXX - Shouldn't this be a shared utility method someplace?
     *
     * @param path Context-relative request path to be checked
     *  (must start with '/')
     * @param pattern URL pattern to be compared against
     */
    private boolean matchPattern(String path, String pattern) {

	// Normalize the argument strings
	if ((path == null) || (path.length() == 0))
	    path = "/";
	if ((pattern == null) || (pattern.length() == 0))
	    pattern = "/";

	// Check for exact match
	if (path.equals(pattern))
	    return (true);

	// Check for universal mapping
	if (pattern.equals("/"))
	    return (true);

	// Check for path prefix matching
	if (pattern.startsWith("/") && pattern.endsWith("/*")) {
	    pattern = pattern.substring(0, pattern.length() - 2);
	    if (pattern.length() == 0)
		return (true);	// "/*" is the same as "/"
	    if (path.endsWith("/"))
		path = path.substring(0, path.length() - 1);
	    while (true) {
		if (pattern.equals(path))
		    return (true);
		int slash = path.lastIndexOf('/');
		if (slash <= 0)
		    break;
		path = path.substring(0, slash);
	    }
	    return (false);
	}

	// Check for suffix matching
	else if (pattern.startsWith("*.")) {
	    int slash = path.lastIndexOf('/');
	    int period = path.lastIndexOf('.');
	    if ((slash >= 0) && (period > slash) &&
		path.endsWith(pattern.substring(1))) {
		return (true);
	    }
	}

	return (false);

    }


}
