/*
 * $Header: /home/cvspublic/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/loader/FileClassLoader.java,v 1.7 2000/05/15 22:35:28 craigmcc Exp $
 * $Revision: 1.7 $
 * $Date: 2000/05/15 22:35:28 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */


package org.apache.tomcat.loader;


import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import org.apache.tomcat.util.StringManager;


/**
 * Implementation of <b>java.lang.ClassLoader</b> that knows how to load
 * classes from disk directories and JAR files.  It also implements
 * the <code>Reloader</code> interface, to provide automatic reloading
 * support to <code>StandardLoader</code>.
 * <p>
 * This code was partially based on the <code>AdaptiveClassLoader</code>
 * module originally copied from Apache JServ, and used in Tomcat 3.x.
 * However, it does class loading in a different order (webapp first then
 * system classes), and allows the set of associated repositories to be
 * modified at runtime.
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.7 $ $Date: 2000/05/15 22:35:28 $
 */

public final class FileClassLoader
    extends ClassLoader
    implements Reloader{


    // ---------------------------------------------------------- Constructors


    /**
     * Construct a new ClassLoader instance with no defined repositories.
     */
    public FileClassLoader() {

        super();

    }


    /**
     * Construct a new ClassLoader instance with the specified initial list of
     * repositories.  Each element of this list must be a String that contains
     * a pathname to a directory, a JAR file, or a ZIP file.
     *
     * @param repositories Initial list of repositories
     *
     * @exception IllegalArgumentException if one of the specified repositories
     *  does not exist, or is not of the appropriate type
     */
    public FileClassLoader(Enumeration repositories) {

        super();
	while (repositories.hasMoreElements())
	    addRepository((String) repositories.nextElement());

    }


    /**
     * Construct a new ClassLoader instance with the specified initial list of
     * repositories.  Each element of this list must be a String that contains
     * a pathname to a directory, a JAR file, or a ZIP file.
     *
     * @param repositories Initial list of repositories
     *
     * @exception IllegalArgumentException if one of the specified repositories
     *  does not exist, or is not of the appropriate type
     */
    public FileClassLoader(String repositories[]) {

        super();
	for (int i = 0; i < repositories.length; i++)
	    addRepository(repositories[i]);

    }


    // ---------------------------------------------------- Instance Variables


    /**
     * The set of CacheEntries for classes loaded by this class loader,
     * keyed by the fully qualified class name.
     */
    private Hashtable cache = new Hashtable();


    /**
     * The debugging detail level of this component.
     */
    //    private int debug = 2;
    private int debug = 0;


    /**
     * The set of File entries for repositories that are directories,
     * keyed by repository name.
     */
    private Hashtable directories = new Hashtable();


    /**
     * The set of File entries for repositories that are JARs,
     * keyed by repository name.
     */
    private Hashtable jarFiles = new Hashtable();


    /**
     * The set of repositories associated with this class loader.  The order
     * of the elements in this Vector controls the order in which repositories
     * are examined when new classes or resources are requested.
     */
    private String repositories[] = new String[0];


    /**
     * The string manager for this package.
     */
    private static final StringManager sm =
        StringManager.getManager(Constants.Package);


    // ------------------------------------------------------------ Properties


    /**
     * Return the debugging detail level of this component.
     */
    public int getDebug() {

        return (this.debug);

    }


    /**
     * Set the debugging detail level of this component.
     *
     * @param debug The new debugging detail level
     */
    public void setDebug(int debug) {

        this.debug = debug;

    }


    // ------------------------------------------------------- Reloader Methods


    /**
     * Add a new repository to the set of places this ClassLoader can look for
     * classes to be loaded.
     *
     * @param repository Name of a source of classes to be loaded, such as a
     *  directory pathname, a JAR file pathname, or a ZIP file pathname
     *
     * @exception IllegalArgumentException if the specified repository is
     *  invalid or does not exist
     */
    public synchronized void addRepository(String repository) {

	if (debug >= 1)
	    log("addRepository(" + repository + ")");
	for (int i = 0; i < repositories.length; i++) {
	    if (repository.equals(repositories[i])) {
		if (debug >= 1)
		    log("  This repository is already included");
		return;
	    }
	}

	// Validate the existence, readability, and type of this repository
	File file = new File(repository);
	if (!file.exists()) {
	    if (debug >= 1)
		log("  Repository does not exist");
	    throw new IllegalArgumentException
	        (sm.getString("fileClassLoader.exists", repository));
	}
	if (!file.canRead()) {
	    if (debug >= 1)
		log("  Repository cannot be read");
	    throw new IllegalArgumentException
	        (sm.getString("fileClassLoader.canRead", repository));
	}
	if (file.isDirectory()) {
	    if (debug >= 1)
		log("  Repository is a directory");
	    directories.put(repository, file);
	} else {
	    ZipFile jarFile = null;
	    try {
	        jarFile = new ZipFile(file);
		Enumeration entries = jarFile.entries();
		while (entries.hasMoreElements()) {
		    ZipEntry entry = (ZipEntry) entries.nextElement();
		    InputStream is = jarFile.getInputStream(entry);
		    is.close();
		    break;
		}
		jarFile.close();
	    } catch (Throwable t) {
		if (debug >= 1)
		    log("  Problem with this ZIP/JAR file", t);
	        throw new IllegalArgumentException
		    (sm.getString("fileClassLoader.jarFile", repository));
	    }
	    if (debug >= 1)
		log("  Repository is a ZIP/JAR file");
	    jarFiles.put(repository, file);
	}

	// Add this repository to the set we know about
	String results[] = new String[repositories.length + 1];
	for (int i = 0; i < repositories.length; i++)
	    results[i] = repositories[i];
	results[results.length - 1] = repository;
	repositories = results;

    }


    /**
     * Return an enumeration of the current repositories for this class
     * loader.  If there are no repositories, an empty Enumeration is
     * returned.
     */
    public String[] findRepositories() {

        return ((String[]) repositories.clone());

    }


    /**
     * Have one or more classes or resources been modified so that a reload
     * is appropriate?
     */
    public boolean modified() {

	//	if (debug >= 2)
	//	    log("modified()");

	// Build a list of the classes we currently have cached
	Vector temp = new Vector();
	synchronized(cache) {
	    Enumeration keys = cache.keys();
	    while (keys.hasMoreElements())
		temp.addElement(keys.nextElement());
	}

	// Check for modifications to any of the cached classes
	Enumeration classnames = temp.elements();
	while (classnames.hasMoreElements()) {
	    String classname = (String) classnames.nextElement();
	    CacheEntry entry = (CacheEntry) cache.get(classname);
	    if (entry == null)
		continue;
	    if (entry.origin == null)	// System classes cannot be modified
		continue;
	    /*
	    if (debug >= 2)
		log("Check classname=" + classname +
		    ", entry=" +
		    (new java.sql.Timestamp(entry.lastModified)).toString() +
		    ", origin=" +
		    (new java.sql.Timestamp(entry.origin.lastModified())).toString());
	    */
	    if (entry.lastModified != entry.origin.lastModified()) {
		//		if (debug >= 2)
		//		    log("  Class " + classname + " was modified");
		return (true);
	    }
	}

	//	if (debug >= 2)
	//	    log("  No classes were modified");
        return (false);

    }


    /**
     * Remove the specified repository from the set of places this ClassLoader
     * can look for classes to be loaded.  Any classes already loaded from this
     * repository will remain, but no future loads from here will take place.
     *
     * @param repository Name of the repository to remove
     */
    public synchronized void removeRepository(String repository) {

	if (debug >= 1)
	    log("removeRepository(" + repository + ")");
	String results[] = new String[repositories.length - 1];
	int j = 0;
	boolean removed = false;
	for (int i = 0; i < repositories.length; i++) {
	    if (repository.equals(repositories[i]))
		removed = true;
	    else
		results[j++] = repositories[i];
	}
	if (debug >= 1) {
	    if (removed)
		log("  Repository has been removed");
	    else
		log("  Repository was not present");
	}

    }


    // --------------------------------------------------- ClassLoader Methods


    /**
     * Find a resource with a given name.  The return is a URL to the resource.
     * Doing a <code>getContent()</code> on the URL may return an Image, an
     * AudioClip, or an InputStream.
     *
     * @param name The name of the resource, to be used as is
     */
    public URL getResource(String name) {

	//	if (debug >= 2)
	//	    log("getResource(" + name + ")");

	// Search for this resource in all of our repositories
	String repositories[] = findRepositories();
	for (int i = 0; i < repositories.length; i++) {
	    String repository = repositories[i];
	    //	    if (debug >= 2)
	    //		log("  Checking repository " + repository);
	    File directory = (File) directories.get(repository);
	    if (directory != null) {
	        URL theURL =
		  loadResourceFromDirectory(directory, name);
		//		if ((theURL != null) && (debug >= 2))
		//		    log("  Returning directory resource URL " + theURL);
		if (theURL != null)
		    return (theURL);
	    }
	    File jarFile = (File) jarFiles.get(repository);
	    if (jarFile != null) {
	        URL theURL =
		  loadResourceFromJarFile(jarFile, name);
		//		if ((theURL != null) && (debug >= 2))
		//		    log("  Returning JAR/ZIP resource URL " + theURL);
		if (theURL != null)
		    return (theURL);
	    }
	}

        // Load this resource from the system class path
	URL theURL = getSystemResource(name);
	/*
	if (debug >= 2) {
	    if (theURL != null)
		log("  Returning system resource URL " + theURL);
	    else
		log("  Cannot find this resource");
	}
	*/
	return (theURL);

    }


    /**
     * Get an InputStream on a given resource.  Will return <code>null</code>
     * if no resource with this name is found.
     *
     * @param name The name of the resource
     */
    public InputStream getResourceAsStream(String name) {

	//	if (debug >= 2)
	//	    log("getResourceAsStream(" + name + ")");

	// Search for this resource in all of our repositories
	String repositories[] = findRepositories();
	for (int i = 0; i < repositories.length; i++) {
	    String repository = repositories[i];
	    //	    if (debug >= 2)
	    //		log("  Checking repository " + repository);
	    File directory = (File) directories.get(repository);
	    if (directory != null) {
	        InputStream theStream =
		  loadStreamFromDirectory(directory, name);
		//		if ((theStream != null) && (debug >= 2))
		//		    log("  Returning directory resource stream");
		if (theStream != null)
		    return (theStream);
	    }
	    File jarFile = (File) jarFiles.get(repository);
	    if (jarFile != null) {
	        InputStream theStream =
		  loadStreamFromJarFile(jarFile, name);
		//		if ((theStream != null) && (debug >= 2))
		//		    log("  Returning JAR/ZIP resource stream");
		if (theStream != null)
		    return (theStream);
	    }
	}

	// Load this resource from the system class path
	InputStream theStream = getSystemResourceAsStream(name);
	/*
	if (debug >= 2) {
	    if (theStream != null)
		log("  Returning system resource stream");
	    else
		log("  Cannot find this resource");
	}
	*/
	return (theStream);

    }


    /**
     * Requests the class loader to load and resolve a class with the
     * specified name.  The <code>loadClass</code> method is called by the
     * Java Virtual Machine when a class loaded by a class loader first
     * references another class.  Every subclass of <code>ClassLoader</code>
     * must define this method.
     *
     * @param name Name of the desired Class
     * @param resolve <code>true</code> if this Class needs to be resolved.
     *
     * @return The resulting Class, or <code>null</code> if it was not found
     *
     * @exception ClassNotFoundException if the class loader cannot find
     *  a definition for this class
     */
    public Class loadClass(String name, boolean resolve)
        throws ClassNotFoundException {

	//	if (debug >= 2)
	//	    log("loadClass(" + name + ", " + resolve + ")");

	// Handle requests for restricted classes by throwing an exception
	if (restricted(name)) {
	    //	    if (debug >= 2)
	    //	        log("  Rejecting restricted class " + name);
	    throw new ClassNotFoundException
	      (sm.getString("fileClassLoader.restricted", name));
	}

	// Has this class already been loaded?
	CacheEntry entry = (CacheEntry) cache.get(name);
	if (entry != null) {
	    Class theClass = entry.loadedClass;
	    if (resolve)
	        resolveClass(theClass);
	    //	    if (debug >= 2)
	    //		log("  Found the class in our cache");
	    return (theClass);
	}

	// Search for this class in all of our repositories
	if (allowed(name)) {
	    String repositories[] = findRepositories();
	    for (int i = 0; i < repositories.length; i++) {
		String repository = repositories[i];
		//		if (debug >= 2)
		//		    log("  Checking repository " + repository);
	        File directory = (File) directories.get(repository);
	        if (directory != null) {
	            Class theClass =
		      loadClassFromDirectory(directory, name, resolve);
		    //		    if ((theClass != null) && (debug >= 2))
		    //			log("  Loaded from a directory");
		    if (theClass != null)
		        return (theClass);
	        }
		File jarFile = (File) jarFiles.get(repository);
		if (jarFile != null) {
		    Class theClass =
		      loadClassFromJarFile(jarFile, name, resolve);
		    //		    if ((theClass != null) && (debug >= 2))
		    //			log("  Loaded from a JAR/ZIP file");
		    if (theClass != null)
		        return (theClass);
		}
	    }
	}

	// Load this class from the system class path
	//	if (debug >= 2)
	//	    log("  Checking the system class path");
	Class theClass = loadClassFromSystem(name, resolve);
	//	if ((theClass != null) && (debug >= 2))
	//	    log("  Loaded from the system class path");
	if (theClass != null)
	    return (theClass);
	else {
	    //	    if (debug >= 2)
	    //		log("  Cannot find this class");
	    throw new ClassNotFoundException(name);
	}

    }


    /**
     * Return a String representation of this class.
     */
    public String toString() {

	StringBuffer sb = new StringBuffer("FileClassLoader[");
	String repositories[] = findRepositories();
	for (int i = 0; i < repositories.length; i++) {
	    if (i > 0)
		sb.append(File.pathSeparator);
	    sb.append(repositories[i]);
	}
	sb.append("]");
	return (sb.toString());

    }


    // ------------------------------------------------------- Private Methods


    /**
     * Should we allow this class to be loaded from someplace other than
     * the system class path?
     *
     * @param name Name of the class we are considering
     */
    private boolean allowed(String name) {

        if (name.startsWith("java."))
	    return (false);
	else
	    return (true);

    }


    /**
     * Load the specified class from the specified directory, if it exists;
     * otherwise return <code>null</code>.  If found, a cache entry will be
     * automatically added before returning.
     *
     * @param directory File for the directory to be searched
     * @param name Name of the desired class
     * @param resolve <code>true</code> if this Class needs to be resolved
     */
    private Class loadClassFromDirectory(File directory, String name,
					 boolean resolve) {

        // Translate the class name to a filename
        String filename =
	  name.replace('.', File.separatorChar) + ".class";

	// Validate the existence and readability of this class file
	File classFile = new File(directory, filename);
	if (!classFile.exists() || !classFile.canRead())
	    return (null);

	// Load the bytes for this class, and define it
	Class theClass = null;
	byte buffer[] = new byte[(int) classFile.length()];
	InputStream is = null;
	try {
	    is = new BufferedInputStream(new FileInputStream(classFile));
	    is.read(buffer);
	    is.close();
	    is = null;
	    theClass = defineClass(name, buffer, 0, buffer.length);
	    if (resolve)
	        resolveClass(theClass);
	    cache.put(name,
		      new CacheEntry(theClass, classFile,
				     classFile.lastModified()));
	    return (theClass);
	} catch (Throwable t) {
	    if (is != null) {
	        try {
		    is.close();
		} catch (Throwable u) {
		    ;
		}
	    }
	    return (null);
	}

    }


    /**
     * Load the specified class from the specified JAR file, if it exists;
     * otherwise return <code>null</code>.  If found, a cache entry will be
     * automatically added before returning.
     *
     * @param jar JAR file to be searched
     * @param name Name of the desired class
     * @param resolve <code>true</code> if this Class needs to be resolved
     */
    private Class loadClassFromJarFile(File jar, String name,
				       boolean resolve) {

        // Translate the class name to a filename
        String filename =
	  name.replace('.', '/') + ".class";

	// Load the bytes for this class, and define it
	ZipFile jarFile = null;
	ZipEntry jarEntry = null;
	Class theClass = null;
	InputStream is = null;
	try {
	    jarFile = new ZipFile(jar);
	    if (jarFile == null) {
		//		if (debug >= 2)
		//		    log("  Cannot open JAR file " + jar.getAbsolutePath());
		return (null);
	    }
	    jarEntry = jarFile.getEntry(filename);
	    if (jarEntry == null) {
	        jarFile.close();
		return (null);
	    }
	    byte buffer[] = new byte[(int) jarEntry.getSize()];
	    is = new BufferedInputStream(jarFile.getInputStream(jarEntry));
	    is.read(buffer);
	    is.close();
	    theClass = defineClass(name, buffer, 0, buffer.length);
	    if (resolve)
	        resolveClass(theClass);
	    cache.put(name,
		      new CacheEntry(theClass, jar, jar.lastModified()));
	    return (theClass);
	} catch (Throwable t) {
	    if (is != null) {
	        try {
		    is.close();
		} catch (Throwable u) {
		    ;
		}
	    }
	    if (jarFile != null) {
	        try {
		    jarFile.close();
		} catch (Throwable u) {
		    ;
		}
	    }
	    return (null);
	}

    }


    /**
     * Load the specified class from the system class path, if it exists;
     * otherwise return <code>null</code>.  If found, a cache entry will be
     * automatically added before returning.
     *
     * @param name Name of the desired class
     * @param resolve <code>true</code> if this Class needs to be resolved
     */
    private Class loadClassFromSystem(String name, boolean resolve) {

        Class theClass = null;
	try {
	    theClass = findSystemClass(name);
	} catch (ClassNotFoundException e) {
	    return (null);
	}
	if (theClass != null) {
	    if (resolve)
	        resolveClass(theClass);
	    cache.put(name,
		      new CacheEntry(theClass, null, Long.MAX_VALUE));
	    return (theClass);
	} else
	    return (null);

    }


    /**
     * Load the specified resource from the specified directory, if it exists;
     * otherwise return <code>null</code>.
     *
     * @param directory File for the directory to be searched
     * @param name Name of the desired resource
     */
    private URL loadResourceFromDirectory(File directory, String name) {

        // Translate the resource name to a filename
	String filename = name.replace('/', File.separatorChar);

	// Validate the existence and readability of this resource file
	File resourceFile = new File(directory, filename);
	if (!resourceFile.exists() || !resourceFile.canRead())
	    return (null);

	// Return a URL that points to this resource file
	try {
	    return (new URL("file:" + resourceFile.getAbsolutePath()));
	} catch (MalformedURLException e) {
	    return (null);
	}

    }


    /**
     * Load the specified resource from the specified JAR file, if it exists;
     * otherwise return <code>null</code>.
     *
     * @param jar JAR file to be searched
     * @param name Name of the desired resource
     */
    private URL loadResourceFromJarFile(File jar, String name) {

        // Translate the resource name to a filename
	//        String filename = name.replace('.', '/');
	String filename = name;

	// Create an input stream for this resource
	ZipFile jarFile = null;
	ZipEntry jarEntry = null;
	InputStream is = null;
	try {
	    jarFile = new ZipFile(jar);
	    jarEntry = jarFile.getEntry(filename);
	    jarFile.close();
	    if (jarEntry == null)
		return (null);
	    try {
		return (new URL("jar:file:" + jar.getAbsolutePath() +
				"!/" + name));
	    } catch (MalformedURLException e) {
		return (null);
	    }
	} catch (Throwable t) {
	    if (jarFile != null) {
	        try {
		    jarFile.close();
		} catch (Throwable u) {
		    ;
		}
	    }
	    return (null);
	}

    }


    /**
     * Load the specified resource from the specified directory, if it exists;
     * otherwise return <code>null</code>.
     *
     * @param directory File for the directory to be searched
     * @param name Name of the desired resource
     */
    private InputStream loadStreamFromDirectory(File directory, String name) {

        // Translate the resource name to a filename
	String filename = name.replace('/', File.separatorChar);

	// Validate the existence and readability of this resource file
	File resourceFile = new File(directory, filename);
	if (!resourceFile.exists() || !resourceFile.canRead())
	    return (null);

	// Return an input stream for this resource file
	try {
	    return (new FileInputStream(resourceFile));
	} catch (IOException e) {
	    return (null);
	}

    }


    /**
     * Load the specified resource from the specified JAR file, if it exists;
     * otherwise return <code>null</code>.
     *
     * @param directory File for the directory to be searched
     * @param name Name of the desired resource
     */
    private InputStream loadStreamFromJarFile(File jar, String name) {

        // Translate the resource name to a filename
	//        String filename = name.replace('.', '/');
	String filename = name;

	// Create an input stream for this resource
	ZipFile jarFile = null;
	ZipEntry jarEntry = null;
	InputStream is = null;
	try {
	    jarFile = new ZipFile(jar);
	    jarEntry = jarFile.getEntry(filename);
	    if (jarEntry == null) {
	        jarFile.close();
		return (null);
	    }
	    is = new BufferedInputStream(jarFile.getInputStream(jarEntry));
	    jarFile.close();
	    return (is);
	} catch (Throwable t) {
	    if (jarFile != null) {
	        try {
		    jarFile.close();
		} catch (Throwable u) {
		    ;
		}
	    }
	    return (null);
	}

    }


    /**
     * Log a debugging output message.
     *
     * @param message Message to be logged
     */
    private void log(String message) {

	System.out.println("FileClassLoader: " + message);

    }


    /**
     * Log a debugging output message with an exception.
     *
     * @param message Message to be logged
     * @param throwable Exception to be logged
     */
    private void log(String message, Throwable throwable) {

	System.out.println("FileClassLoader: " + message);
	throwable.printStackTrace(System.out);

    }


    /**
     * Is this a class that should not be allowed in a web application?
     *
     * @param name Name of the class to be checked
     */
    private boolean restricted(String name) {

	return (name.startsWith("org.apache.tomcat."));

    }


    // ------------------------------------------------------- Private Classes


    /**
     * The cache entry for a particular class loaded by this class loader.
     */
    private static class CacheEntry {

        /**
	 * The "last modified" time of the origin file at the time this class
	 * was loaded, in milliseconds since the epoch.
	 */
        long lastModified;

        /**
	 * The actual loaded class.
	 */
        Class loadedClass;

        /**
	 * The File (for a directory) or JarFile (for a JAR) from which this
	 * class was loaded, or <code>null</code> if loaded from the system.
	 */
        File origin;

        /**
	 * Construct a new instance of this class.
	 */
        public CacheEntry(Class loadedClass, File origin,
			  long lastModified) {

	    this.loadedClass = loadedClass;
	    this.origin = origin;
	    this.lastModified = lastModified;

	}

    }


}
