/*
 * $Header: /home/cvspublic/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/resources/Attic/StandardResources.java,v 1.10 2000/06/19 21:46:50 craigmcc Exp $
 * $Revision: 1.10 $
 * $Date: 2000/06/19 21:46:50 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */


package org.apache.tomcat.resources;


import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Vector;
import org.apache.tomcat.Container;
import org.apache.tomcat.Context;
import org.apache.tomcat.Host;
import org.apache.tomcat.Lifecycle;
import org.apache.tomcat.LifecycleEvent;
import org.apache.tomcat.LifecycleException;
import org.apache.tomcat.LifecycleListener;
import org.apache.tomcat.Logger;
import org.apache.tomcat.Resources;
import org.apache.tomcat.util.LifecycleSupport;
import org.apache.tomcat.util.StringManager;


/**
 * Standard implementation of the <b>Resources</b> interface.  Resource
 * requests are resolved against the document base that is configured.
 * For the purposes of this implementation, a document base can be:
 * <ul>
 * <li>An absolute disk directory pathname (no trailing slash).
 * <li>A relative disk directory pathname (no trailing slash), which will
 *     be resolved against the current working directory at runtime
 * <li>A URL prefix (no trailing slash) to a resource base that is potentially
 *     remote.
 * </ul>
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.10 $ $Date: 2000/06/19 21:46:50 $
 */

public final class StandardResources
    implements Resources, Lifecycle, PropertyChangeListener {


    // ----------------------------------------------------------- Constructors


    /**
     * Construct a new instance of this class with default values.
     */
    public StandardResources() {

	super();

    }


    // ----------------------------------------------------- Instance Variables


    /**
     * The Container this component is associated with.
     */
    private Container container = null;


    /**
     * The debugging detail level for this component.
     */
    private int debug = 0;


    /**
     * The document root for this component.
     */
    private String docBase = null;


    /**
     * The document root for this component, expressed as an absolute or
     * relative pathname, or <code>null</code> if the document root is
     * not filesystem-based.
     */
    private String fileBase = null;


    /**
     * The descriptive information string for this implementation.
     */
    private static final String info =
	"org.apache.tomcat.resources.StandardResources/1.0";


    /**
     * The lifecycle event support for this component.
     */
    protected LifecycleSupport lifecycle = new LifecycleSupport(this);


    /**
     * The string manager for this package.
     */
    private static final StringManager sm =
	StringManager.getManager(Constants.Package);


    /**
     * Has this component been started?
     */
    private boolean started = false;


    /**
     * The property change support for this component.
     */
    protected PropertyChangeSupport support = new PropertyChangeSupport(this);


    // ------------------------------------------------------------- Properties


    /**
     * Return the Container with which this Resources has been associated.
     */
    public Container getContainer() {

	return (this.container);

    }


    /**
     * Set the Container with which this Resources has been associated.
     *
     * @param container The associated Container
     */
    public void setContainer(Container container) {

	Container oldContainer = this.container;
	if ((oldContainer != null) && (oldContainer instanceof Context))
	    ((Context) oldContainer).removePropertyChangeListener(this);

	this.container = container;
	// We are interested in property changes to the document base
	if ((this.container != null) && (this.container instanceof Context)) {
	    ((Context) this.container).addPropertyChangeListener(this);
	    setDocBase(((Context) this.container).getDocBase());
	}

	support.firePropertyChange("container", oldContainer, this.container);

    }


    /**
     * Return the debugging detail level for this component.
     */
    public int getDebug() {

	return (this.debug);

    }


    /**
     * Set the debugging detail level for this component.
     *
     * @param debug The new debugging detail level
     */
    public void setDebug(int debug) {

	this.debug = debug;

    }


    /**
     * Return the document root for this component.
     */
    public String getDocBase() {

	return (this.docBase);

    }


    /**
     * Set the document root for this component.
     *
     * @param docBase The new document root
     *
     * @exception IllegalArgumentException if this would create a
     *  malformed URL
     */
    public void setDocBase(String docBase) {

	// Validate the format of the proposed document root
	if (docBase == null)
	    throw new IllegalArgumentException
		(sm.getString("standardResources.null"));
	if ((docBase.length() > 1) && docBase.endsWith("/"))
	    throw new IllegalArgumentException
		(sm.getString("standardResources.slash", docBase));

	// Change the document root property
	String oldDocBase = this.docBase;
	this.docBase = docBase.toString();
	support.firePropertyChange("docBase", oldDocBase, this.docBase);
	if (debug >= 1)
	    log("Setting docBase to '" + this.docBase + "'");

	// Calculate the corresponding file base (if any)
	fileBase = null;
	int colon = this.docBase.indexOf(":");
	if (colon <= 1)			// No colon, or Windows drive letter
	    fileBase = this.docBase;
	if (this.docBase.startsWith("file:")) {
	    if (this.docBase.startsWith("file://"))
		fileBase = this.docBase.substring(7);
	    else
		fileBase = this.docBase.substring(5);
	    if ((fileBase.length() > 2) &&
		(fileBase.charAt(1) == '|'))
		fileBase =
		    fileBase.substring(0, 1) + ":" + fileBase.substring(2);
	}

	// Make sure that fileBase is an absolute pathname to a directory
	if (fileBase != null) {
	    File file = new File(fileBase);
	    if (!file.isAbsolute()) {
		file = new File(hostBase(), fileBase);
		fileBase = file.getAbsolutePath();
	    }
	    if (debug >= 1)
		log(" Also setting fileBase to '" + fileBase + "'");
	    if (!file.isDirectory())
		throw new IllegalArgumentException
		    (sm.getString("standardResources.directory", fileBase));
	    if (!file.exists())
	    	throw new IllegalArgumentException
	    	    (sm.getString("standardResources.exists", fileBase));
	}


    }


    /**
     * Return descriptive information about this Resources implementation and
     * the corresponding version number, in the format
     * <code>&lt;description&gt;/&lt;version&gt;</code>.
     */
    public String getInfo() {

	return (info);

    }


    // --------------------------------------------------------- Public Methods


    /**
     * Add a property change listener to this component.
     *
     * @param listener The listener to add
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {

	support.addPropertyChangeListener(listener);

    }


    /**
     * Return the MIME type of the specified file, or <code>null</code> if
     * the MIME type is not known.  The MIME type is determined by the
     * configuration of the servlet container, and may be specified in a
     * web application descriptor.  Common MIME types are
     * <code>"text/html"</code> and <code>"image/gif"</code>.
     *
     * @param file Name of the file whose MIME type is to be determined
     */
    public String getMimeType(String file) {

	if (debug >= 1)
	    log("Calculating MIME type of '" + file + "'");
	if (file == null)
	    return (null);
	if ((container == null) || (!(container instanceof Context)))
	    return (null);
	int period = file.lastIndexOf(".");
	if (period < 0)
	    return (null);
	String extension = file.substring(period + 1);
	if (extension.length() < 1)
	    return (null);
	if (debug >= 1)
	    log(" Mime type of '" + extension + "' is '" +
		((Context) container).findMimeMapping(extension));
	return (((Context) container).findMimeMapping(extension));

    }


    /**
     * Return the real path for a given virtual path.  For example, the
     * virtual path <code>"/index.html"</code> has a real path of whatever
     * file on the server's filesystem would be served by a request for
     * <code>"/index.html"</code>.
     * <p>
     * The real path returned will be in a form appropriate to the computer
     * and operating system on which the servlet container is running,
     * including the proper path separators.  This method returns
     * <code>null</code> if the servlet container cannot translate the
     * virtual path to a real path for any reason (such as when the content
     * is being made available from a <code>.war</code> archive).
     *
     * @param path The virtual path to be translated
     */
    public String getRealPath(String path) {

	if ((fileBase == null) || (path == null) ||
	    (!path.startsWith("/") && !path.startsWith("\\"))) {
	    if (debug >= 2)
	    	log("getRealPath(" + path + ") --> null");
	    return (null);
	} else {
	    if (debug >= 2)
	        log("getRealPath(" + path + ") --> " + fileBase + path);
	    return (fileBase + path);
	}

    }


    /**
     * Return a URL to the resource that is mapped to the specified path.
     * The path must begin with a "/" and is interpreted as relative to
     * the current context root.
     * <p>
     * This method allows the Container to make a resource available to
     * servlets from any source.  Resources can be located on a local or
     * remote file system, in a database, or in a <code>.war</code> file.
     * <p>
     * The servlet container must implement the URL handlers and
     * <code>URLConnection</code> objects that are necessary to access
     * the resource.
     * <p>
     * This method returns <code>null</code> if no resource is mapped to
     * the pathname.
     * <p>
     * Some Containers may allow writing to the URL returned by this method,
     * using the methods of the URL class.
     * <p>
     * The resource content is returned directly, so be aware that
     * requesting a <code>.jsp</code> page returns the JSP source code.
     * Use a <code>RequestDispatcher</code> instead to include results
     * of an execution.
     * <p>
     * This method has a different purpose than
     * <code>java.lang.Class.getResource()</code>, which looks up resources
     * based on a class loader.  This method does not use class loaders.
     *
     * @param path The path to the desired resource
     *
     * @exception MalformedURLException if the pathname is not given
     *  in the correct form
     */
    public URL getResource(String path) throws MalformedURLException {

	if (path == null)
	    return (null);
	if (path.indexOf("/..") >= 0)
	    return (null);  // Refuse to interpret relative paths

	// Deal with URL-based document root
	if (fileBase == null) {
	    ;	// FIXME -- return NULL if resource does not exist
	    return (new URL(docBase + path));
	}

	// Deal with file-based document root
	File file = new File(fileBase + path);
	if (!file.exists())
	    return (null);
	String url = null;
	if ((fileBase.length() > 2) && (fileBase.charAt(1) == ':'))
	    url = "file:///" + fileBase.substring(0, 1) + ":" +
		fileBase.substring(2) + path;
	else
	    url = "file://" + fileBase + path;
	return (new URL(url));

    }


    /**
     * Return the resource located at the named path as an
     * <code>InputStream</code> object.
     * <p>
     * The data in the <code>InputStream</code> can be of any type or length.
     * The path must be specified according to the rules given in
     * <code>getResource()</code>.  This method returns <code>null</code>
     * if no resource exists at the specified path.
     * <p>
     * Meta-information such as content length and content type that is
     * available via the <code>getResource()</code> method is lost when
     * using this method.
     * <p>
     * The servlet container must implement the URL handlers and
     * <code>URLConnection</code> objects that are necessary to access
     * the resource.
     * <p>
     * This method is different from
     * <code>java.lang.Class.getResourceAsStream()</code>, which uses a
     * class loader.  This method allows servlet containers to make a
     * resource available to a servlet from any location, without using
     * a class loader.
     *
     * @param path The path to the desired resource
     */
    public InputStream getResourceAsStream(String path) {

	if (path == null) {
	    if (debug >= 2)
		log("getResourceAsStream(" + path +
		    ") --> null due to null argument");
	    return (null);
	}
	if (path.indexOf("/..") >= 0) {
	    if (debug >= 2)
		log("getResourceAsStream(" + path +
		    ") --> null due to '/..' in path");
	    return (null);  // Refuse to interpret relative paths
	}

	// Deal with URL-based document root
	if (fileBase == null) {
	    try {
		URL url = getResource(path);
		if (debug >= 2)
		    log("getResourceAsStream(" + path +
			") --> URL " + url.toString());
		return (url.openStream());
	    } catch (MalformedURLException e) {
		if (debug >= 2)
		    log("getResourceAsStream(" + path +
			") --> null due to MalformedURLException", e);
		return (null);
	    } catch (IOException e) {
		if (debug >= 2)
		    log("getResourceAsStream(" + path +
			") --> null due to IOException", e);
		return (null);
	    }
	}

	// Deal with file-based document root
	try {
	    if (debug >= 2)
		log("getResourceAsStream(" + path +
		    ") --> FileInputStream(" + fileBase + path + ")");
	    return (new FileInputStream(fileBase + path));
	} catch (Throwable t) {
	    if (debug >= 2)
		log("getResourceAsStream(" + path +
		    ") --> null due to Throwable", t);
	    return (null);
	}

    }


    /**
     * Return the last modified date/time of the resource at the specified
     * path, where <code>path</code> would be suitable for passing as an
     * argument to <code>getResource()</code> or
     * <code>getResourceAsStream()</code>.  If there is no resource at the
     * specified location, return -1.
     *
     * @param path The path to the desired resource
     */
    public long getResourceModified(String path) {

	if (path == null)
	    return (-1);

	// Deal with URL-based document root
	if (fileBase == null) {
	    return (-1);	// FIXME - URL-based document root
	}

	// Deal with file-based document root
	File file = new File(fileBase + path);
	if (file.exists())
	    return (file.lastModified());
	else
	    return (-1);

    }


    /**
     * Remove a property change listener from this component.
     *
     * @param listener The listener to remove
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {

	support.removePropertyChangeListener(listener);

    }


    // ----------------------------------------- PropertyChangeListener Methods


    /**
     * Process property change events from our associated Context.
     *
     * @param event The property change event that has occurred
     */
    public void propertyChange(PropertyChangeEvent event) {

	// Validate the source of this event
	if (!(event.getSource() instanceof Context))
	    return;
	Context context = (Context) event.getSource();

	// Process a relevant property change
	if (event.getPropertyName().equals("docBase"))
	    setDocBase((String) event.getNewValue());

    }


    // -------------------------------------------------------- Private Methods


    /**
     * Return a File object representing the base directory for the
     * entire servlet container (i.e. the Engine container if present).
     */
    private File engineBase() {

	return (new File(System.getProperty("catalina.home")));

    }


    /**
     * Return a File object representing the base directory for the
     * current virtual host (i.e. the Host container if present).
     */
    private File hostBase() {

	// Locate our immediately surrounding Host (if any)
	Container container = this.container;
	while (container != null) {
	    if (container instanceof Host)
		break;
	    container = container.getParent();
	}
	if (container == null)
	    return (engineBase());

	// Use the "appBase" property of this container
	String appBase = ((Host) container).getAppBase();
	File file = new File(appBase);
	if (!file.isAbsolute())
	    file = new File(engineBase(), appBase);
	return (file);

    }



    /**
     * Log a message on the Logger associated with our Container (if any)
     *
     * @param message Message to be logged
     */
    private void log(String message) {

	Logger logger = null;
	if (container != null)
	    logger = container.getLogger();
	if (logger != null)
	    logger.log("StandardResources[" + container.getName() + "]: "
		       + message);
	else {
	    String containerName = null;
	    if (container != null)
		containerName = container.getName();
	    System.out.println("StandardResources[" + containerName
			       + "]: " + message);
	}

    }


    /**
     * Log a message on the Logger associated with our Container (if any)
     *
     * @param message Message to be logged
     * @param throwable Associated exception
     */
    private void log(String message, Throwable throwable) {

	Logger logger = null;
	if (container != null)
	    logger = container.getLogger();
	if (logger != null)
	    logger.log("StandardResources[" + container.getName() + "] "
		       + message, throwable);
	else {
	    String containerName = null;
	    if (container != null)
		containerName = container.getName();
	    System.out.println("StandardResources[" + containerName
			       + "]: " + message);
	    System.out.println("" + throwable);
	    throwable.printStackTrace(System.out);
	}

    }


    // ------------------------------------------------------ Lifecycle Methods


    /**
     * Add a lifecycle event listener to this component.
     *
     * @param listener The listener to add
     */
    public void addLifecycleListener(LifecycleListener listener) {

	lifecycle.addLifecycleListener(listener);

    }


    /**
     * Remove a lifecycle event listener from this component.
     *
     * @param listener The listener to remove
     */
    public void removeLifecycleListener(LifecycleListener listener) {

	lifecycle.removeLifecycleListener(listener);

    }


    /**
     * Prepare for the beginning of active use of the public methods of this
     * component.  This method should be called after <code>configure()</code>,
     * and before any of the public methods of the component are utilized.
     *
     * @exception IllegalStateException if this component has already been
     *  started
     * @exception LifecycleException if this component detects a fatal error
     *  that prevents this component from being used
     */
    public void start() throws LifecycleException {

	// Validate and update our current component state
	if (started)
	    throw new LifecycleException
		(sm.getString("standardResources.alreadyStarted"));
	lifecycle.fireLifecycleEvent(START_EVENT, null);
	started = true;

    }


    /**
     * Gracefully terminate the active use of the public methods of this
     * component.  This method should be the last one called on a given
     * instance of this component.
     *
     * @exception IllegalStateException if this component has not been started
     * @exception LifecycleException if this component detects a fatal error
     *  that needs to be reported
     */
    public void stop() throws LifecycleException {

	// Validate and update our current state
	if (!started)
	    throw new LifecycleException
		(sm.getString("standardResources.notStarted"));

	lifecycle.fireLifecycleEvent(STOP_EVENT, null);
	started = false;

    }


}
