/*
 * $Header: /home/cvspublic/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/servlets/DefaultServlet.java,v 1.9 2000/06/24 19:48:56 remm Exp $
 * $Revision: 1.9 $
 * $Date: 2000/06/24 19:48:56 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */


package org.apache.tomcat.servlets;


import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.sql.Timestamp;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;
import java.util.StringTokenizer;
import java.util.Locale;
import java.util.Hashtable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.ServletContext;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.tomcat.util.MD5Encoder;
import org.apache.tomcat.util.StringManager;
import org.apache.tomcat.util.xml.SaxContext;
import org.apache.tomcat.util.xml.XmlAction;
import org.apache.tomcat.util.xml.XmlMapper;


/**
 * The default resource-serving servlet for most web applications,
 * used to serve static resources such as HTML pages and images.
 *
 * @author Craig R. McClanahan
 * @author Remy Maucherat
 * @version $Revision: 1.9 $ $Date: 2000/06/24 19:48:56 $
 */

public final class DefaultServlet
    extends HttpServlet {


    // -------------------------------------------------------------- Constants


    /**
     * Maximum file size for which content is cached. 32 ko by default.
     */
    public static final long MAX_SIZE_CACHE = 32768;


    /**
     * Cache entry timeout. 10s by default.
     */
    public static final long CACHE_ENTRY_TIMEOUT = 10000;


    // ----------------------------------------------------- Instance Variables


    /**
     * The debugging detail level for this servlet.
     */
    private int debug = 0;


    /**
     * The input buffer size to use when serving resources.
     */
    private int input = 2048;


    /**
     * The output buffer size to use when serving resources.
     */
    private int output = 2048;


    /**
     * The set of welcome files for this web application
     */
    private Vector welcomes = new Vector();


    /**
     * MD5 message digest provider.
     */
    private static MessageDigest md5Helper;


    /**
     * The MD5 helper object for this class.
     */
    private static final MD5Encoder md5Encoder = new MD5Encoder();


    /**
     * The set of SimpleDateFormat formats to use in getDateHeader().
     */
    private static final SimpleDateFormat formats[] = {
	new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz", Locale.US),
	new SimpleDateFormat("EEEEEE, dd-MMM-yy HH:mm:ss zzz", Locale.US),
	new SimpleDateFormat("EEE MMMM d HH:mm:ss yyyy", Locale.US)
    };


    /**
     * MIME multipart separation string
     */
    private static final String mimeSeparation = "THIS_STRING_SEPARATES";


    /**
     * File descriptor cache
     */
    private Hashtable fileCache = new Hashtable();


    /**
     * The string manager for this package.
     */
    private static StringManager sm =
	StringManager.getManager(Constants.Package);


    // --------------------------------------------------------- Public Methods


    /**
     * Finalize this servlet.
     */
    public void destroy() {

	;	// No actions necessary

    }


    /**
     * Process a GET request for the specified resource.
     *
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet-specified error occurs
     */
    public void doGet(HttpServletRequest request,
		      HttpServletResponse response)
	throws IOException, ServletException {

	// Serve the requested resource, including the data content
	serveResource(request, response, true);

    }


    /**
     * Process a HEAD request for the specified resource.
     *
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet-specified error occurs
     */
    public void doHead(HttpServletRequest request,
		       HttpServletResponse response)
	throws IOException, ServletException {

	// Serve the requested resource, without the data content
	serveResource(request, response, false);

    }


    /**
     * Process a POST request for the specified resource.
     *
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet-specified error occurs
     */
    public void doPost(HttpServletRequest request,
		       HttpServletResponse response)
	throws IOException, ServletException {

	doGet(request, response);

    }


    /**
     * Initialize this servlet.
     */
    public void init() throws ServletException {

	// Set our properties from the initialization parameters
	String value = null;
	try {
	    value = getServletConfig().getInitParameter("debug");
	    debug = Integer.parseInt(value);
	} catch (Throwable t) {
	    ;
	}
	try {
	    value = getServletConfig().getInitParameter("input");
	    input = Integer.parseInt(value);
	} catch (Throwable t) {
	    ;
	}
	try {
	    value = getServletConfig().getInitParameter("output");
	    output = Integer.parseInt(value);
	} catch (Throwable t) {
	    ;
	}

	// Sanity check on the specified buffer sizes
	if (input < 256)
	    input = 256;
	if (output < 256)
	    output = 256;

	// Initialize the set of welcome files for this application
	welcomes = (Vector) getServletContext().getAttribute
	    ("org.apache.tomcat.WELCOME_FILES");
	if (welcomes == null)
	    welcomes = new Vector();

	if (debug > 0) {
	    log("DefaultServlet.init:  input buffer size=" + input +
		", output buffer size=" + output);
	    for (int i = 0; i < welcomes.size(); i++)
		log("DefaultServlet.init:  welcome file=" +
		    welcomes.elementAt(i));
	}

        // Load the MD5 helper used to calculate signatures.
        try {
            md5Helper = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
            throw new IllegalStateException();
        }

    }



    // -------------------------------------------------------- Private Methods


    /**
     * Copy the contents of the specified input stream to the specified
     * output stream, and ensure that both streams are closed before returning
     * (even in the face of an exception).
     *
     * @param istream The input stream to read from
     * @param ostream The output stream to write to
     *
     * @exception IOException if an input/output error occurs
     */
    private void copy(CacheEntry fileInfo, ServletOutputStream ostream)
	throws IOException {

        IOException exception = null;
        
        if (fileInfo.content == null) {
            
            // FIXME : i18n ?
            InputStream istream = new BufferedInputStream
                (new FileInputStream(fileInfo.file), input);
            
            // Copy the input stream to the output stream
            if (fileInfo.length > 0)
                exception = copyRange(istream, ostream);
            
            // Clean up the input and output streams
            try {
                istream.close();
            } catch (Throwable t) {
                ;
            }
            
        } else {
            
            try {
                ostream.write(fileInfo.content);
            } catch (IOException e) {
                exception = e;
            }
            
        }
        
	try {
	    ostream.flush();
	} catch (Throwable t) {
	    ;
	}
	try {
	    ostream.close();
	} catch (Throwable t) {
	    ;
	}

	// Rethrow any exception that has occurred
	if (exception != null)
	    throw exception;

    }


    /**
     * Copy the contents of the specified input stream to the specified
     * output stream, and ensure that both streams are closed before returning
     * (even in the face of an exception).
     *
     * @param file The file object
     * @param ostream The output stream to write to
     * @param range Range the client wanted to retrieve
     * @exception IOException if an input/output error occurs
     */
    private void copy(File file, ServletOutputStream ostream, Range range)
	throws IOException {
        
        IOException exception = null;
        
        InputStream istream =
            new BufferedInputStream(new FileInputStream(file), input);
        exception = copyRange(istream, ostream, range.start, range.end);
        
	// Clean up the input and output streams
	try {
	    istream.close();
	} catch (Throwable t) {
	    ;
	}
	try {
	    ostream.flush();
	} catch (Throwable t) {
	    ;
	}
	try {
	    ostream.close();
	} catch (Throwable t) {
	    ;
	}

	// Rethrow any exception that has occurred
	if (exception != null)
	    throw exception;
        
    }


    /**
     * Copy the contents of the specified input stream to the specified
     * output stream, and ensure that both streams are closed before returning
     * (even in the face of an exception).
     *
     * @param file The file object
     * @param ostream The output stream to write to
     * @param ranges Enumeration of the ranges the client wanted to retrieve
     * @param contentType Content type of the resource
     * @exception IOException if an input/output error occurs
     */
    private void copy(File file, ServletOutputStream ostream,
                      Enumeration ranges, String contentType)
	throws IOException {
        
        IOException exception = null;
        
        while ( (exception == null) && (ranges.hasMoreElements()) ) {
            
            InputStream istream =	// FIXME: internationalization???????
                new BufferedInputStream(new FileInputStream(file), input);
        
            Range currentRange = (Range) ranges.nextElement();
            
            // Writing MIME header.
            ostream.println("--" + mimeSeparation);
            if (contentType != null)
                ostream.println("Content-Type: " + contentType);
            ostream.println("Content-Range: bytes " + currentRange.start
                           + "-" + currentRange.end + "/" 
                           + currentRange.length);
            ostream.println();
            
            // Printing content
            exception = copyRange(istream, ostream, currentRange.start,
                                  currentRange.end);
            
            ostream.println();
            ostream.println();
            
            try {
                istream.close();
            } catch (Throwable t) {
                ;
            }
            
        }
        
        ostream.print("--" + mimeSeparation + "--");
        
	// Clean up the output streams
	try {
	    ostream.flush();
	} catch (Throwable t) {
	    ;
	}
	try {
	    ostream.close();
	} catch (Throwable t) {
	    ;
	}

	// Rethrow any exception that has occurred
	if (exception != null)
	    throw exception;
        
    }


    /**
     * Copy the contents of the specified input stream to the specified
     * output stream, and ensure that both streams are closed before returning
     * (even in the face of an exception).
     *
     * @param istream The input stream to read from
     * @param ostream The output stream to write to
     * @return Exception which occured during processing
     */
    private IOException copyRange(InputStream istream, 
                                  ServletOutputStream ostream) {
        
	// Copy the input stream to the output stream
	IOException exception = null;
	byte buffer[] = new byte[input];
	int len = buffer.length;
	while (len >= buffer.length) {
	    try {
		len = istream.read(buffer);
		ostream.write(buffer, 0, len);
	    } catch (IOException e) {
		exception = e;
		len = -1;
	    }
	    if (len < buffer.length)
		break;
	}
        return exception;
        
    }


    /**
     * Copy the contents of the specified input stream to the specified
     * output stream, and ensure that both streams are closed before returning
     * (even in the face of an exception).
     *
     * @param istream The input stream to read from
     * @param ostream The output stream to write to
     * @param start Start of the range which will be copied
     * @param end End of the range which will be copied
     * @return Exception which occured during processing
     */
    private IOException copyRange(InputStream istream, 
                                  ServletOutputStream ostream,
                                  long start, long end) {
        
        try {
            istream.skip(start);
        } catch (IOException e) {
            return e;
        }
        
	IOException exception = null;
        long bytesToRead = end - start + 1;
        
	byte buffer[] = new byte[input];
	int len = buffer.length;
	while ( (bytesToRead > 0) && (len >= buffer.length)) {
	    try {
                len = istream.read(buffer);
                if (bytesToRead >= len) {
                    ostream.write(buffer, 0, len);
                    bytesToRead -= len; 
                } else {
                    ostream.write(buffer, 0, (int) bytesToRead);
                    bytesToRead = 0;
                }
	    } catch (IOException e) {
		exception = e;
		len = -1;
	    }
	    if (len < buffer.length)
		break;
	}
        
        return exception;
        
    }


    /**
     * Serve the specified directory as a set of hyperlinks to the
     * included files.
     *
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     * @param directory File object representing this directory
     * @param content Should the content be included?
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet-specified error occurs
     */
    private void serveDirectory(HttpServletRequest request,
                                HttpServletResponse response,
                                File directory,
                                boolean content)
	throws IOException, ServletException {

	if (!content)
	    return;

	// Remember the servlet path that got us here
	String servletPath = request.getServletPath();
	if (servletPath == null)
	    servletPath = "/";

	// Redirect if this path does not end with a trailing slash
	// (Otherwise, relative references will not work correctly)
	if (!servletPath.endsWith("/")) {
	    String path = request.getContextPath();
	    if (path == null)
		path = "";
	    path += servletPath + "/";
	    response.sendRedirect(path);
	    return;
	}

	// Serve a welcome resource or file if one exists
	// FIXME - update the welcome files list?
	for (int i = 0; i < welcomes.size(); i++) {
	    
	    // Does the specified resource exist?
	    File file = new File(directory, (String) welcomes.elementAt(i));
	    if (!file.exists() || !file.canRead() || !file.isFile())
		continue;

	    // Can we dispatch a request for this resource (i.e. JSP page)?
	    String resource = servletPath + (String) welcomes.elementAt(i);
	    RequestDispatcher rd =
		getServletContext().getRequestDispatcher(resource);
	    if (rd != null) {
		rd.forward(request, response);
		return;
	    }

	    // Can we serve a file for this resource?
	    serveFile(request, response, file.getAbsolutePath(), content);
	    return;

	}

	// FIXME - do the directory thing
	response.setContentType("text/html");
	if (!content)
	    return;
	response.setBufferSize(output);
	PrintWriter writer = response.getWriter();
        
        printDirectoryContents(request, directory, writer);
        
	writer.flush();
	writer.close();

    }


    /**
     * Print directory content.
     * 
     * @param request Servlet request
     * @param directory File object representing this directory
     * @param writer Writer object
     *
     * @exception IOException if an input/output error occurs
     */
    private void printDirectoryContents(HttpServletRequest request, 
                                        File directory, 
                                        PrintWriter writer)
        throws IOException {
        
	String pathInfo = request.getPathInfo();
        String absPath = request.getPathTranslated();
	String requestURI = request.getRequestURI();
	if (pathInfo == null)
            pathInfo = requestURI;
        
        StringBuffer buf = new StringBuffer();
	
        buf.append("<html>\r\n");
        buf.append("<head>\r\n");
        buf.append("<title>")
            .append(sm.getString("defaultservlet.directorylistingfor"))
            .append(requestURI);
        buf.append("</title>\r\n</head><body bgcolor=white>\r\n");

	buf.append("<table width=90% cellspacing=0 ");
	buf.append("cellpadding=5 align=center>");
	buf.append("<tr><td colspan=3><font size=+2><strong>");
	buf.append(sm.getString("defaultservlet.directorylistingfor"))
	    .append(requestURI);
	buf.append("</strong></td></tr>\r\n");

	if (! pathInfo.equals("/")) {
	    buf.append("<tr><td colspan=3 bgcolor=#ffffff>");
	    //buf.append("<a href=\"../\">Up one directory");
	    
	    String toPath = requestURI;

	    if (toPath.endsWith("/")) {
		toPath = toPath.substring(0, toPath.length() - 1);
	    }
	    
	    toPath = toPath.substring(0, toPath.lastIndexOf("/"));
	    
	    if (toPath.length() == 0) {
		toPath = "/";
	    }
	    
	    buf.append("<a href=\"" + toPath + "\"><tt>"+
		       sm.getString("defaultservlet.upto")+ toPath);
	    buf.append("</tt></a></td></tr>\r\n");
	}

	// Pre-calculate the request URI for efficiency

	// Make another URI that definitely ends with a /
	String slashedRequestURI = null;

	if (requestURI.endsWith("/")) {
	    slashedRequestURI = requestURI;
	} else {
	    slashedRequestURI = requestURI + "/";
	}

	String[] fileNames = directory.list();
	boolean dirsHead=true;
	boolean shaderow = false;

	for (int i = 0; i < fileNames.length; i++) {
	    String fileName = fileNames[i];

            // Don't display special dirs at top level
	    if( "/".equals(pathInfo) &&
		"WEB-INF".equalsIgnoreCase(fileName) ||
		"META-INF".equalsIgnoreCase(fileName) )
		continue;
	    
	    File f = new File(directory, fileName);

	    if (f.isDirectory()) {
		if( dirsHead ) {
		    dirsHead=false;
		    buf.append("<tr><td colspan=3 bgcolor=#cccccc>");
		    buf.append("<font size=+2><strong>").
			append( sm.getString("defaultservlet.subdirectories")).
			append( "</strong>\r\n");
		    buf.append("</font></td></tr>\r\n");
		}

                String fileN = f.getName();

                buf.append("<tr");

                if (shaderow) buf.append(" bgcolor=#eeeeee");
		shaderow = !shaderow;
		
                buf.append("><td>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
                buf.append("<tt><a href=\"")
		    .append(slashedRequestURI)
                    .append(fileN)
		    .append("\">")
		    .append(fileN)
                    .append("/</a>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;")
                    .append("</tt>\r\n");
                buf.append("</td><td><tt>&nbsp;&nbsp;</tt></td>");
                buf.append("<td align=right><tt>");
		buf.append(formats[0].format(new Date(f.lastModified())));
                buf.append("</tt></td></tr>\r\n");
	    }
	}

	shaderow = false;
	buf.append("<tr><td colspan=3 bgcolor=#ffffff>&nbsp;</td></tr>");
	boolean fileHead=true;
	
	for (int i = 0; i < fileNames.length; i++) {
	    File f = new File(directory, fileNames[i]);

	    if (f.isFile()) {
		String fileN = f.getName();
		
		if( fileHead ) {
		    fileHead=false;
		    buf.append("<tr><td colspan=4 bgcolor=#cccccc>");
		    buf.append("<font size=+2><strong>")
			.append(sm.getString("defaultservlet.files"))
			.append("</strong></font></td></tr>");
		}

		buf.append("<tr");

		if (shaderow) buf.append(" bgcolor=#eeeeee");
		shaderow = ! shaderow;
		
		buf.append("><td>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\r\n");

		buf.append("<tt><a href=\"")
		    .append(slashedRequestURI)
		    .append(fileN).append("\">")
		    .append( fileN )
		    .append( "</a>");
		buf.append("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</tt>");
		buf.append("</td>\r\n");

		buf.append("<td align=right><tt>");
		displaySize( buf, (int)f.length());
		buf.append("</tt></td>");

		buf.append("<td align=right><tt>");
		buf.append(formats[0].format(new Date(f.lastModified())));
		buf.append("</tt></td></tr>\r\n");
	    }
	    
	    buf.append("\r\n");
	}
	
	buf.append("<tr><td colspan=3 bgcolor=#ffffff>&nbsp;</td></tr>");
	buf.append("<tr><td colspan=3 bgcolor=#cccccc>");
	buf.append("<font size=-1>");
	buf.append(Constants.TOMCAT_NAME);
	buf.append(" v");
	buf.append(Constants.TOMCAT_VERSION);
	buf.append("</font></td></tr></table>");
	
        buf.append("</body></html>\r\n");
        
        writer.print(buf);
        
    }


    /**
     * Display the size of a file.
     */
    private void displaySize(StringBuffer buf, int filesize) {
        
	int leftside = filesize / 1024;
	int rightside = (filesize % 1024) / 103;  // makes 1 digit
	// To avoid 0.0 for non-zero file, we bump to 0.1
	if (leftside == 0 && rightside == 0 && filesize != 0) 
	    rightside = 1;
	buf.append(leftside).append(".").append(rightside);
	buf.append(" KB");
        
    }
    
    
    /**
     * Check to see if a default page exists.
     * 
     * @param pathname Pathname of the file to be served
     */
    private CacheEntry checkWelcomeFiles(String pathname) {
        
        if (pathname.endsWith("/")) {
            
            // Serve a welcome resource or file if one exists
            // FIXME - update the welcome files list?
            for (int i = 0; i < welcomes.size(); i++) {
                
                // Does the specified resource exist?
                String fileName = pathname + welcomes.elementAt(i);
                CacheEntry fileInfo = (CacheEntry) fileCache.get(fileName);
                if ((fileInfo != null) && (fileInfo.isValid())) {
                    return fileInfo;
                }
                
            }
            
        }
        
        return null;
        
    }


    /**
     * Serve the specified file, optionally including the data content.
     *
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     * @param pathname Pathname of the file to be served
     * @param content Should the content be included?
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet-specified error occurs
     */
    private void serveFile(HttpServletRequest request,
                           HttpServletResponse response,
                           String pathname,
                           boolean content)
	throws IOException, ServletException {

        // Do a lookup in the cache
        CacheEntry fileInfo = (CacheEntry) fileCache.get(pathname);
        
        if (fileInfo == null) {
            fileInfo = checkWelcomeFiles(pathname);
        }
        
        if (fileInfo == null) {
            
            // Open the file (if it actually exists)
            File file = new File(pathname);
            if (!file.exists() || !file.canRead()) {
                response.sendError(HttpServletResponse.SC_NOT_FOUND, pathname);
                return;
            }
            if (file.isDirectory()) {
                serveDirectory(request, response, file, content);
                return;
            }
            if (debug > 0)
                log("DefaultServlet.serveFile: Serving '" + pathname + "'");
            
            fileInfo = new CacheEntry(pathname);
            fileCache.put(pathname, fileInfo);
            
        } else {
            
            //System.out.println("Cache hit : " + pathname);
            
            if (!fileInfo.isValid()) {
                
                //System.out.println("Cache invalid");
                
                if (!fileInfo.revalidate()) {
                    
                    fileInfo = null;
                    
                    // Open the file (if it actually exists)
                    File file = new File(pathname);
                    if (!file.exists() || !file.canRead()) {
                        response.sendError
                            (HttpServletResponse.SC_NOT_FOUND, pathname);
                        return;
                    }
                    if (file.isDirectory()) {
                        serveDirectory(request, response, file, content);
                        return;
                    }
                    if (debug > 0)
                        log("DefaultServlet.serveFile: Serving '" 
                            + pathname + "'");
                    
                    fileInfo = new CacheEntry(pathname);
                    fileCache.put(pathname, fileInfo);
                    
                }
            }

        }


        // Checking If headers
        if ( !checkIfHeaders(request, response, fileInfo) )
            return;
        
        // Parse range specifier
        Vector ranges = parseRange(request, response, fileInfo);
        
        // Last-Modified header
        if (debug > 0)
            log("DefaultServlet.serveFile:  lastModified='" +
                (new Timestamp(fileInfo.date)).toString() + "'");
        response.setDateHeader("Last-Modified", fileInfo.date);
        
        // ETag header
        response.setHeader("ETag", getETag(fileInfo, true));
        
        // Find content type.
        String contentType = getServletContext().getMimeType(pathname);
        
        if ( (ranges == null) && (request.getHeader("Range") == null) ) {
            
            // Set the appropriate output headers
            if (contentType != null) {
                if (debug > 0)
                    log("DefaultServlet.serveFile:  contentType='" +
                        contentType + "'");
                response.setContentType(contentType);
            }
            long contentLength = fileInfo.length;
            if (contentLength >= 0) {
                if (debug > 0)
                    log("DefaultServlet.serveFile:  contentLength=" +
                        contentLength);
                response.setContentLength((int) contentLength);
            }
            
            // Copy the input stream to our output stream (if requested)
            if (content) {
                response.setBufferSize(output);
                copy(fileInfo, response.getOutputStream());
            }
            
        } else {
            
            if (ranges == null)
                return;
            
            // Partial content response.
            
            response.setStatus(HttpServletResponse.SC_PARTIAL_CONTENT);
            
            if (ranges.size() == 1) {
                
                Range range = (Range) ranges.elementAt(0);
                response.addHeader("Content-Range", "bytes " 
                                   + range.start
                                   + "-" + range.end + "/" 
                                   + range.length);
                
                if (contentType != null) {
                    if (debug > 0)
                        log("DefaultServlet.serveFile:  contentType='" +
                            contentType + "'");
                    response.setContentType(contentType);
                }
                
                if (content) {
                    response.setBufferSize(output);
                    copy(fileInfo.file, response.getOutputStream(), range);
                }
                
            } else {
                
                response.setContentType("multipart/byteranges; boundary="
                                        + mimeSeparation);
                
                if (content) {
                    response.setBufferSize(output);
                    copy(fileInfo.file, response.getOutputStream(), 
                         ranges.elements(), contentType);
                }
                
            }
            
        }
        
    }


    /**
     * Serve the specified resource, optionally including the data content.
     *
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     * @param content Should the content be included?
     *
     * @exception IOException if an input/output error occurs
     * @exception ServletException if a servlet-specified error occurs
     */
    private void serveResource(HttpServletRequest request,
                               HttpServletResponse response,
                               boolean content)
	throws IOException, ServletException {

	// Identify the requested resource path
	String servletPath = request.getServletPath();
	if (servletPath == null)
	    servletPath = "/";
	if (debug > 0) {
	    if (content)
		log("DefaultServlet.serveResource:  Serving resource '" +
		    servletPath + "' headers and data");
	    else
		log("DefaultServlet.serveResource:  Serving resource '" +
		    servletPath + "' headers only");
	}
	if (servletPath == null) {
	    response.sendError(HttpServletResponse.SC_BAD_REQUEST);
	    return;
	}

	// Exclude any resource in the /WEB-INF subdirectory
	if (servletPath.startsWith("/WEB-INF")) {
	    response.sendError(HttpServletResponse.SC_NOT_FOUND, servletPath);
	    return;
	}

	// Convert the resource path to a URL
	URL resourceURL = null;
	try {
	    resourceURL = getServletContext().getResource(servletPath);
	} catch (MalformedURLException e) {
	    ;
	}
	if (resourceURL == null) {
	    response.sendError(HttpServletResponse.SC_NOT_FOUND, servletPath);
	    return;
	}
	if (debug > 0)
	    log("DefaultServlet.serveResource:  Corresponding URL is '" +
		resourceURL.toString() + "'");

	// Optimized processing for a "file:" URL
	if ("file".equals(resourceURL.getProtocol())) {
	    serveFile(request, response, resourceURL.getFile(), content);
	    return;
	}

    }


    /**
     * Check if the conditions specified in the optional If headers are 
     * satisfied.
     * 
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     * @param fileInfo File object
     * @return boolean true if the resource meets all the specified conditions,
     * and false if any of the conditions is not satisfied, in which case
     * request processing is stopped
     */
    private boolean checkIfHeaders(HttpServletRequest request,
                                   HttpServletResponse response, 
                                   CacheEntry fileInfo)
        throws IOException {
        
        String eTag = getETag(fileInfo, true);
        long fileLength = fileInfo.length;
        long lastModified = fileInfo.date;
        
        StringTokenizer commaTokenizer;
        
        String headerValue;
        
        // Checking If-Match
        headerValue = request.getHeader("If-Match");
        if (headerValue != null) {
            if (headerValue.indexOf("*") == -1) {
                
                commaTokenizer = new StringTokenizer(headerValue, ",");
                boolean conditionSatisfied = false;
                
                while (!conditionSatisfied && commaTokenizer.hasMoreTokens()) {
                    String currentToken = commaTokenizer.nextToken();
                    if (currentToken.trim().equals(eTag))
                        conditionSatisfied = true;
                }
                
                // If none of the given ETags match, 412 Precodition failed is
                // sent back
                if (!conditionSatisfied) {
                    response.sendError
                        (HttpServletResponse.SC_PRECONDITION_FAILED);
                    return false;
                }
                
            }
        }
        
        // Checking If-Modified-Since
        headerValue = request.getHeader("If-Modified-Since");
        if (headerValue != null) {
            
            // If an If-None-Match header has been specified, if modified since
            // is ignored.
            if (request.getHeader("If-None-Match") == null) {
                
                Date date = null;
                
                // Parsing the HTTP Date
                for (int i = 0; (date == null) && (i < formats.length); i++) {
                    try {
                        date = formats[i].parse(headerValue);
                    } catch (ParseException e) {
                        ;
                    }
                }
                
                if ((date != null) 
                    && (lastModified <= (date.getTime() + 1000)) ) {
                    // The entity has not been modified since the date 
                    // specified by the client. This is not an error case.
                    response.sendError
                        (HttpServletResponse.SC_NOT_MODIFIED);
                    return false;
                }
                
            }
            
        }
        
        // Checking If-None-Match
        headerValue = request.getHeader("If-None-Match");
        if (headerValue != null) {
            if (headerValue.indexOf("*") == -1) {
                
                commaTokenizer = new StringTokenizer(headerValue, ",");
                boolean conditionSatisfied = false;
                
                while (!conditionSatisfied && commaTokenizer.hasMoreTokens()) {
                    String currentToken = commaTokenizer.nextToken();
                    if (currentToken.trim().equals(eTag))
                        conditionSatisfied = true;
                }
                
                if (conditionSatisfied) {
                    
                    // For GET and HEAD, we should respond with 
                    // 304 Not Modified.
                    // For every other method, 412 Precondition Failed is sent
                    // back.
                    if ( ("GET".equals(request.getMethod()))
                         || ("HEAD".equals(request.getMethod())) ) {
                        response.sendError
                            (HttpServletResponse.SC_NOT_MODIFIED);
                        return false;
                    } else {
                        response.sendError
                            (HttpServletResponse.SC_PRECONDITION_FAILED);
                        return false;
                    }
                }
                
            } else {
                if (fileInfo.file.exists()) {
                    
                }
            }
        }
        
        // Checking If-Unmodified-Since
        headerValue = request.getHeader("If-Unmodified-Since");
        if (headerValue != null) {
            
            Date date = null;
            
            // Parsing the HTTP Date
            for (int i = 0; (date == null) && (i < formats.length); i++) {
                try {
                    date = formats[i].parse(headerValue);
                } catch (ParseException e) {
                    ;
                }
            }
            
            if ( (date != null) && (lastModified > date.getTime()) ) {
                // The entity has not been modified since the date 
                // specified by the client. This is not an error case.
                response.sendError
                    (HttpServletResponse.SC_PRECONDITION_FAILED);
                return false;
            }
            
        }
        
        return true;
    }


    /**
     * Get the ETag value associated with a file.
     * 
     * @param fileInfo File object
     * @param strong True if we want a strong ETag, in which case a checksum
     * of the file has to be calculated
     */
    private String getETagValue(CacheEntry fileInfo, boolean strong) {
        // FIXME : Compute a strong ETag if requested, using an MD5 digest
        // of the file contents
        return fileInfo.length + "-" + fileInfo.date;
    }


    /**
     * Get the ETag associated with a file.
     * 
     * @param fileInfo File object
     * @param strong True if we want a strong ETag, in which case a checksum
     * of the file has to be calculated
     */
    private String getETag(CacheEntry fileInfo, boolean strong) {
        if (strong)
            return "\"" + getETagValue(fileInfo, strong) + "\"";
        else
            return "W/\"" + getETagValue(fileInfo, strong) + "\"";
    }


    /**
     * Parse the range header.
     * 
     * @param request The servlet request we are processing
     * @param response The servlet response we are creating
     * @return Vector of ranges
     */
    private Vector parseRange(HttpServletRequest request, 
                              HttpServletResponse response, 
                              CacheEntry fileInfo) 
        throws IOException {
        
        // Checking If-Range
        String headerValue = request.getHeader("If-Range");
        if (headerValue != null) {
            
            String eTag = getETag(fileInfo, true);
            long lastModified = fileInfo.date;
            
            Date date = null;
            
            // Parsing the HTTP Date
            for (int i = 0; (date == null) && (i < formats.length); i++) {
                try {
                    date = formats[i].parse(headerValue);
                } catch (ParseException e) {
                    ;
                }
            }
            
            if (date == null) {
                
                // If the ETag the client gave does not match the entity
                // etag, then the entire entity is returned.
                if (!eTag.equals(headerValue.trim()))
                    return null;
                
            } else {
                
                // If the timestamp of the entity the client got is older than
                // the last modification date of the entity, the entire entity
                // is returned.
                if (lastModified > (date.getTime() + 1000))
                    return null;
                
            }
            
        }
        
        long fileLength = fileInfo.length;
        
        if (fileLength == 0)
            return null;
        
        // Retrieving the range header (if any is specified
        String rangeHeader = request.getHeader("Range");
        
        if (rangeHeader == null)
            return null;
        // bytes is the only range unit supported (and I don't see the point
        // of adding new ones).
        if (!rangeHeader.startsWith("bytes")) {
            response.sendError
                (HttpServletResponse.SC_REQUESTED_RANGE_NOT_SATISFIABLE);
            return null;
        }
        
        rangeHeader = rangeHeader.substring(6);
        
        // Vector which will contain all the ranges which are successfully
        // parsed.
        Vector result = new Vector();
        StringTokenizer commaTokenizer = new StringTokenizer(rangeHeader, ",");
        
        // Parsing the range list
        while (commaTokenizer.hasMoreTokens()) {
            String rangeDefinition = commaTokenizer.nextToken();
            
            Range currentRange = new Range();
            currentRange.length = fileLength;
            
            int dashPos = rangeDefinition.indexOf('-');
            
            if (dashPos == -1) {
                response.sendError
                    (HttpServletResponse.SC_REQUESTED_RANGE_NOT_SATISFIABLE);
                return null;
            }
            
            if (dashPos == 0) {
                
                try {
                    long offset = Long.parseLong(rangeDefinition);
                    currentRange.start = fileLength + offset;
                    currentRange.end = fileLength - 1;
                } catch (NumberFormatException e) {
                    response.sendError
                        (HttpServletResponse
                         .SC_REQUESTED_RANGE_NOT_SATISFIABLE);
                    return null;
                }
                
            } else {
                
                try {
                    currentRange.start = Long.parseLong
                        (rangeDefinition.substring(0, dashPos));
                    if (dashPos < rangeDefinition.length() - 1)
                        currentRange.end = Long.parseLong
                            (rangeDefinition.substring
                             (dashPos + 1, rangeDefinition.length()));
                    else
                        currentRange.end = fileLength - 1;
                } catch (NumberFormatException e) {
                    response.sendError
                        (HttpServletResponse
                         .SC_REQUESTED_RANGE_NOT_SATISFIABLE);
                    return null;
                }
                
            }
            
            if (!currentRange.validate()) {
                response.sendError
                    (HttpServletResponse.SC_REQUESTED_RANGE_NOT_SATISFIABLE);
                return null;
            }
            
            result.addElement(currentRange);
        }
        
        return result;
    }


    // ------------------------------------------------------ Range Inner Class


    private class Range {
        
        public long start;
        public long end;
        public long length;
        
        /**
         * Validate range.
         */
        public boolean validate() {
            return ( (start >= 0) && (end >= 0) && (length > 0)
                     && (start <= end) && (end < length) );
        }
        
    }


    // ------------------------------------------------  CacheEntry Inner Class


    private class CacheEntry {


        /**
         * Constructor.
         * 
         * @param pathname Path name of the file
         */
        public CacheEntry(String fileName) 
            throws IOException {
            
            this.validUntil = System.currentTimeMillis() + CACHE_ENTRY_TIMEOUT;
            this.fileName = fileName;
            this.file = new File(fileName);
            this.date = file.lastModified();
            this.httpDate = formats[0].format(new Date(date));
            this.length = file.length();
            
            // Copying the file contents into the cache, if and only if the
            // file size is smaller than the amount specified
            if (this.length < MAX_SIZE_CACHE) {
                content = new byte[(int) this.length];
                InputStream istream = new FileInputStream(file);
                
                int pos = 0;
                while (pos < this.length) {
                    pos += istream.read(content, pos, (int) this.length - pos);
                }
            }
            
        }


        public long validUntil;
        public String fileName;
        public File file;
        public String httpDate;
        public long date;
        public long length;
        public byte[] content;


        /**
         * Is the cache entry still valid ?
         */
        public boolean isValid() {
            return (System.currentTimeMillis() <= validUntil);
        }


        /**
         * Revalidate a cache entry.
         */
        public boolean revalidate() {
            if ((file.length() == length) && (file.lastModified() == date)) {
                validUntil = System.currentTimeMillis() + CACHE_ENTRY_TIMEOUT;
                return true;
            } else {
                return false;
            }
        }


        /**
         * String representation.
         */
        public String toString() {
            return fileName;
        }


    }


}
