/*
 * $Header: /home/cvs/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/core/StandardLoader.java,v 1.2 2000/02/09 05:10:08 craigmcc Exp $
 * $Revision: 1.2 $
 * $Date: 2000/02/09 05:10:08 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 


package org.apache.tomcat.core;


import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Vector;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import org.apache.tomcat.Container;
import org.apache.tomcat.Lifecycle;
import org.apache.tomcat.LifecycleException;
import org.apache.tomcat.Loader;
import org.apache.tomcat.loader.AdaptiveClassLoader;
import org.apache.tomcat.util.StringManager;


/**
 * Standard implementation of the <b>Loader</b> interface that wraps the
 * <code>AdaptiveClassLoader</code> implementation imported from the
 * Apache JServ project.  This class loader supports detection of modified
 * Java classes, which can be used to implement auto-reload support.
 * <p>
 * This class loader is configured by adding the pathnames of directories,
 * JAR files, and ZIP files with the <code>addRepository()</code> method,
 * prior to calling <code>start()</code>.  When a new class is required,
 * these repositories will be consulted first to locate the class.  If it
 * is not present, the system class loader will be used instead.
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.2 $ $Date: 2000/02/09 05:10:08 $
 */

public final class StandardLoader
    implements Lifecycle, Loader {


    // ----------------------------------------------------- Instance Variables


    /**
     * The class loader being managed by this Loader component.
     */
    private AdaptiveClassLoader classLoader = null;


    /**
     * The Container with which this Loader has been associated.
     */
    private Container container = null;


    /**
     * The descriptive information about this Loader implementation.
     */
    private static final String info =
	"org.apache.tomcat.core.StandardLoader/1.0";


    /**
     * The set of class repositories for this class loader.  The contents of
     * this list must be directories, JAR files, and ZIP files that contain
     * the classes to be loaded by this class loader.
     */
    private Vector repositories = new Vector();


    /**
     * The string manager for this package.
     */
    protected StringManager sm =
	StringManager.getManager(Constants.Package);


    /**
     * Has this component been started?
     */
    private boolean started = false;


    /**
     * The property change support for this component.
     */
    protected PropertyChangeSupport support = new PropertyChangeSupport(this);


    // ------------------------------------------------------------- Properties


    /**
     * Return the Java class loader to be used by this Container.
     */
    public ClassLoader getClassLoader() {

	return (classLoader);

    }


    /**
     * Return the Container with which this Logger has been associated.
     */
    public Container getContainer() {

	return (container);

    }


    /**
     * Set the Container with which this Logger has been associated.
     *
     * @param container The associated Container
     */
    public void setContainer(Container container) {

	Container oldContainer = this.container;
	this.container = container;
	support.firePropertyChange("container", oldContainer, this.container);

    }


    /**
     * Return descriptive information about this Loader implementation and
     * the corresponding version number, in the format
     * <code>&lt;description&gt;/&lt;version&gt;</code>.
     */
    public String getInfo() {

	return (info);

    }


    // --------------------------------------------------------- Public Methods


    /**
     * Add a property change listener to this component.
     *
     * @param listener The listener to add
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {

	support.addPropertyChangeListener(listener);

    }


    /**
     * Add a new repository to the set of repositories for this class loader.
     *
     * @param repository Repository to be added
     */
    public void addRepository(String repository) {

	if (!repositories.contains(repository))
	    repositories.addElement(repository);

    }


    /**
     * Return the set of repositories defined for this class loader.
     * If none are defined, a zero-length array is returned.
     */
    public String[] findRepositories() {

	synchronized (repositories) {
	    String results[] = new String[repositories.size()];
	    for (int i = 0; i < results.length; i++)
		results[i] = (String) repositories.elementAt(i);
	    return (results);
	}

    }


    /**
     * Has the internal repository associated with this Loader been modified,
     * such that the loaded classes should be reloaded?
     */
    public boolean modified() {

	if (classLoader != null)
	    return (classLoader.shouldReload());
	else
	    return (false);

    }


    /**
     * Cause the underlying class loader (and therefore all of the classes
     * loaded by that class loader) to be thrown away, and creates a new one.
     *
     * @exception IllegalStateException if a change to the repositories for
     *  this class loader has rendered restart impossible
     */
    public void reload() {

	try {
	    stop();
	    start();
	} catch (Throwable t) {
	    throw new IllegalStateException("reload: " + t);
	}

    }


    /**
     * Remove a property change listener from this component.
     *
     * @param listener The listener to remove
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {

	support.removePropertyChangeListener(listener);

    }


    /**
     * Remove a repository from the set of repositories for this loader.
     *
     * @param repository Repository to be removed
     */
    public void removeRepository(String repository) {

	repositories.removeElement(repository);

    }


    // ------------------------------------------------------ Lifecycle Methods


    /**
     * Start this component, initializing our associated class loader.
     *
     * @exception LifecycleException if a lifecycle error occurs
     */
    public void start() throws LifecycleException {

	// Validate and update our current component state
	if (started)
	    throw new LifecycleException
		(sm.getString("standardLoader.alreadyStarted"));
	started = true;

	// Validate the specified repositories
	Vector files = new Vector();
	Enumeration names = repositories.elements();
	while (names.hasMoreElements()) {
	    File file = new File((String) names.nextElement());
	    files.addElement(file);
	    String path = null;
	    try {
		path = file.getCanonicalPath();
	    } catch (IOException e) {
		path = "UNKNOWN";
	    }
	    if (!file.exists())
		throw new LifecycleException
		    (sm.getString("standardLoader.exists", path));
	    else if (!file.canRead())
		throw new LifecycleException
		    (sm.getString("standardLoader.read", path));
	    else if (!checkType(file))
		throw new LifecycleException
		    (sm.getString("standardLoader.type", path));
	}

	// Construct a new class loader instance for our use
	try {
	    classLoader = new AdaptiveClassLoader(files);
	} catch (IllegalArgumentException e) {
	    throw new LifecycleException("start: ", e);
	}

    }


    /**
     * Stop this component, finalizing our associated class loader.
     *
     * @exception LifecycleException if a lifecycle error occurs
     */
    public void stop() throws LifecycleException {

	// Validate and update our current component state
	if (!started)
	    throw new LifecycleException
		(sm.getString("standardLoader.notStarted"));
	started = false;

	// Throw away our current class loader
	classLoader = null;

    }


    // ------------------------------------------------------- Private Methods


    /**
     * Is the specified file an appropriate entry for a repository?
     *
     * @param file The repository entry to be validated
     */
    private boolean checkType(File file) {

	// Directories are acceptable
	if (file.isDirectory())
	    return (true);

	// ZIP and JAR files that we can recognize are acceptable
	boolean isArchive = true;
	ZipFile zipFile = null;
	try {
	    zipFile = new ZipFile(file);
	} catch (ZipException e) {
	    isArchive = false;
	} catch (IOException e) {
	    isArchive = false;
	}
	if (zipFile != null) {
	    try {
		zipFile.close();
	    } catch (IOException e) {
		;
	    }
	}
	return (isArchive);

    }


}

