/*
 * $Header: /home/cvs/jakarta-tomcat/proposals/catalina/src/share/org/apache/tomcat/security/HttpBasicAuth.java,v 1.2 2000/02/13 01:43:47 craigmcc Exp $
 * $Revision: 1.2 $
 * $Date: 2000/02/13 01:43:47 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights 
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:  
 *       "This product includes software developed by the 
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written 
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */ 


package org.apache.tomcat.security;


import java.io.IOException;
import java.security.Principal;
import javax.servlet.http.HttpServletResponse;
import org.apache.tomcat.Realm;
import org.apache.tomcat.Request;
import org.apache.tomcat.Response;
import org.apache.tomcat.deploy.LoginConfig;


/**
 * Utility methods that support HTTP Basic Authentication for the
 * <code>SecurityInterceptor</code> implementation.
 *
 * @author Craig R. McClanahan
 * @version $Revision: 1.2 $ $Date: 2000/02/13 01:43:47 $
 */

final class HttpBasicAuth {


    // --------------------------------------------------------- Public Methods


    /**
     * Authenticate the user making this request, using HTTP BASIC
     * authentication (see RFC 2617).  Return <code>true</code> if the
     * user has already been authenticated successfully, or
     * <code>false</code> if we have issued an authentication challenge.
     *
     * @param request Request we are processing
     * @param response Response we are creating
     * @param login LoginConfig describing how authentication
     *  should be performed
     * @param realm Realm used to authenticate individual users
     *
     * @exception IOException if an input/output error occurs
     */
    public static boolean authenticate(Request request, Response response,
				       LoginConfig config, Realm realm)
	throws IOException {

	// Validate any credentials already included with this request
	String authorization = request.getRequest().getHeader("Authorization");
	if (authorization != null) {
	    Principal principal = findPrincipal(authorization, realm);
	    if (principal != null) {
		request.setUserPrincipal(principal);
		return (true);
	    }
	}

	// Send an "unauthorized" response and an appropriate challenge
	String realmName = config.getRealmName();
	if (realmName == null)
	    realmName = request.getRequest().getServerName() + ":" +
		request.getRequest().getServerPort();
	response.getResponse().setHeader
	    ("WWW-Authenticate", "Basic \"" + realmName + "\"");
	response.getResponse().setStatus(HttpServletResponse.SC_UNAUTHORIZED);
	response.flush();
	return (false);

    }


    /**
     * Parse the specified authorization credentials, and return the
     * associated Principal that these credentials authenticate (if any)
     * from the specified Realm.  If there is no such Principal, return
     * <code>null</code>.
     *
     * @param authorization Authorization credentials from this request
     * @param realm Realm used to authenticate Principals
     */
    public static Principal findPrincipal(String authorization, Realm realm) {

	// Validate the authorization credentials format
	if (authorization == null)
	    return (null);
	if (!authorization.startsWith("Basic "))
	    return (null);
	authorization = authorization.substring(6).trim();

	// Decode and parse the authorization credentials
	String unencoded = authorization;	// XXX - Base64 Decoder needed!
	int colon = unencoded.indexOf(':');
	if (colon < 0)
	    return (null);
	String username = unencoded.substring(0, colon);
	String password = unencoded.substring(colon + 1);

	// Validate these credentials in our associated realm
	return (realm.authenticate(username, password));

    }


}
