#include <time.h>
#include <string.h>
#include <stdlib.h>
#include "Server.h"
#include "responses.h"
#include "defines.h"

void CServer::Connect(const char *hostname, unsigned short port)
{
    char buf[BUFSIZE];

    log.Log("Connecting to NNTP server\n");
    if(!sock.Connect(hostname, port))
        log.Abort(CCNSERVER, "Error connecting to NNTP"
            " server, %s:%d\n", hostname, port);
    GetResponse();
}

void CServer::Disconnect()
{
    sock.Disconnect();
}

int CServer::GetResponse(char *userbuf, int bufsize)
{
    static char buf[BUFSIZE];
    if(!sock.Receive(buf))
        log.Abort(NSERVERERROR, "NNTP server timed out.\n");

    if(userbuf)
	strncpy(userbuf, buf, bufsize);

    int resplist_size = sizeof(nntp_responses)/sizeof(nntp_responses[0]);
    int i;
    for(i=0; i<resplist_size &&
        strncmp(buf, nntp_responses[i].code, 3); i++);

    if(i==resplist_size)
    {
        log.Abort(NSERVERERROR, "Unrecognized NNTP response: %s\n",
                  buf);
    }
    else
    {
        if(nntp_responses[i].type == RT_OK)
	    return nntp_responses[i].exitcode;
        else if(nntp_responses[i].type == RT_WARN)
        {
            log.Warn("NNTP server: %s\n", buf);
            return nntp_responses[i].exitcode;
        }
        else if(nntp_responses[i].type == RT_ABORT)
        {
            log.Abort(nntp_responses[i].exitcode,
                      "NNTP server: %s\n", buf);
            return 0;    // Never gets here anyway
        }
    }
}

int CServer::SetGroup(CGroup& group)
{
    char buf[BUFSIZE];
    int count = 0;

    msgids.Clear();

    log.Log("Checking group %s\n", group.groupname);
    sock.Send("GROUP %s\n", group.groupname);
    if(!GetResponse(buf, BUFSIZE))
        return 0;

    if(group.type == BY_ARTNUM)
    {
	int num, oldest, newest;
	char *ptr = strtok(buf+4, " ");
	num = atoi(ptr);
	ptr = strtok(NULL, " ");
	oldest = atoi(ptr);
	ptr = strtok(NULL, " ");
	newest = atoi(ptr);

	int artnum = atoi(group.artnum);
	if(artnum != -1)
	{
	    for(int i=artnum+1; i<=newest; i++)
		msgids << i << endl;
	    count = newest - artnum;
	}
	sprintf(group.artnum, "%d", newest);
    }
    else if(group.type == BY_DATE)
    {
	time_t ltime;
	struct tm *ltm;
	char dbuf[10], tbuf[10];

        time(&ltime);
        ltm = localtime(&ltime);
        strftime(dbuf, 9, "%y%m%d", ltm);
        strftime(tbuf, 9, "%H%M%S", ltm);

	sock.Send("NEWNEWS %s %s %s\n", group.groupname,
		  group.date, group.time);
	if(!GetResponse())
	    return 0;
	sock.Receive(buf);
	while(strcmp(buf, "."))
	{
	    count++;
	    msgids << buf << endl;
	    sock.Receive(buf);
	}
        strcpy(group.date, dbuf);
        strcpy(group.time, tbuf);
    }

    msgids.Rewind();
    log.Log("%d new message(s).\n", count);
    return count;
}

int CServer::GetNextMessage(CMessage& msg)
{
    msgids.peek();
    if(msgids.eof())
        return 0;

    char msgid[BUFSIZE], buf[BUFSIZE];
    int counter = 0, resp;

    msg.Init();

    msgids.getline(msgid, BUFSIZE-1);
    sock.Send("HEAD %s\n", msgid);
    if(!(resp=GetResponse()))
	return 0;
    if(resp == OK_NOSUCHART)
	return OK_NOSUCHART;
    sock.Receive(buf);
    while(strcmp(buf, "."))
    {
        msg.head << buf << endl;
        sock.Receive(buf);
    }

    msg.GetHeader("From: ", buf);
    log.Log("Message from %s", buf);

    sock.Send("BODY %s\n", msgid);
    if(!GetResponse())
	return 0;
    sock.Receive(buf);
    while(strcmp(buf, "."))
    {
        msg.body << buf << endl;
        if(counter++%10 == 0)
            log.Log(".");
        sock.Receive(buf);
    }
    log.Log("\n");
    return 1;
}

int CServer::GetGroupList(ostream& str)
{
    char buf[BUFSIZE];
    int counter = 0;

    log.Log("Getting list of groups");
    sock.Send("LIST\n");
    GetResponse();
    sock.Receive(buf);
    while(strcmp(buf, "."))
    {
        char *ptr = strchr(buf, ' ');
        if(ptr) *ptr = '\0';
        str << buf << endl;
        if(counter++%20 == 0)
            log.Log(".");
        sock.Receive(buf);
    }
    log.Log("\n");
    return 1;
}
