#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "Config.h"
#include "Message.h"
#include "Digest.h"
#include "Server.h"
#include "responses.h"
#include "defines.h"

// Globals

char *info = "news2mail 1.1, by Alex Shnitman <alexsh@linux.org.il>\n\n";
int quiet = 0;
int nolog = 0;
int listonly = 0;
char listfile[BUFSIZE] = "groups.lst";

void ParseArgs(int argc, char *argv[]);
void Usage();

int main(int argc, char *argv[])
{
    ParseArgs(argc, argv);
    if(!quiet)
        cout << info;

    CConfig cfg;
    if(!cfg.ReadConfig(CONFIG_FILE))
    {
        if(quiet<2)
            cerr << "FATAL: Cannot read config file." << endl;
        exit(NOCFGFILE);
    }
    if(nolog && cfg.logfile)
        free(cfg.logfile), cfg.logfile = NULL;

    CLog log;
    log.Init(cfg.logfile);
    log.ShutUp(quiet);
    CServer serv(log);

    if(listonly)
    {
        serv.Connect(cfg.nntpserver, cfg.nntpport);
        ofstream ofs(listfile, ios::out | ios::trunc);
        serv.GetGroupList(ofs);
        ofs.close();
        serv.Disconnect();
        return 0;
    }

    if(!cfg.ReadGroups(GROUPS_FILE))
    {
        if(quiet<2)
            cerr << "FATAL: Cannot read groups file." << endl;
        exit(NOGRPFILE);
    }

    if(!cfg.groups.GetNumItems())
    {
        log.Log("There are no newsgroups to process.\n");
        exit(0);
    }

    serv.Connect(cfg.nntpserver, cfg.nntpport);

    CDigest dig(log, cfg);
    CMessage msg;

    for(int i=0; i<cfg.groups.GetNumItems(); i++)
    {
        CGroup group = cfg.groups.GetAt(i);

        if(!group.emails.GetNumItems())
            log.Log("No recipients for group %s, skipping\n",
                group.groupname);
        else if(serv.SetGroup(group) != 0)
        {
	    int resp, counter = 0;
	    
            dig.Init(group);
            while(resp=serv.GetNextMessage(msg))
		if(resp != OK_NOSUCHART)
		{
		    dig.AddMessage(msg);
		    counter++;
		}
	    log.Log("%d message(s) downloaded.\n", counter);
            dig.SendDigest();
        }

        cfg.WriteGroups(GROUPS_FILE);
    }

    serv.Disconnect();
    return 0;
}


void ParseArgs(int argc, char *argv[])
{
    int jump = 1;
    for(int i=1; i<argc; i+=jump, jump=1)
        if(argv[i][0] = '-')
            for(int j=1; j<strlen(argv[1]); j++)
            switch(argv[i][j])
            {
            case 'q': quiet = 1; break;
            case 'Q': quiet = 2; break;
            case 'n': nolog = 1; break;
            case 'l': listonly = 1; 
                if(i+1<argc && argv[i+1][0] != '-')
                {
                    strncpy(listfile, argv[i+1], BUFSIZE-1);
                    jump++;
                }
                break;
            default: Usage();
            }
        else
            Usage();
}

void Usage()
{
    cerr << "Usage: news2mail [-q|-Q] [-n] [-l [filename]]" << endl
         << "\t-q\tStop the program from putting out log information"
         << endl << "\t\tto the screen."
         << endl
         << "\t-Q\tDon't even output warning messages and fatal errors."
         << endl
         << "\t-n\tDon't log to a log file, even if one is specified in"
         << endl << "\t\tthe configuration file."
         << endl
         << "\t-l\tOnly get the newsgroup list and exit. Write to filename"
         << endl << "\t\tif specified, or to \"groups.lst\" otherwise."
         << endl;
    exit(0);
}
